/*
 * periph_ctl.h
 *
 *  Created on: 10-Feb-2023
 *      Author: jainv
 */

#ifndef PERIPH_CTL_H_
#define PERIPH_CTL_H_

#define C_Ds1307   0x68  // DS1307 ID


#define C_Ds1307SecondRegAddress_U8   0x00u   // Address to access Ds1307 SEC register
#define C_Ds1307DateRegAddress_U8     0x04u   // Address to access Ds1307 DATE register
#define C_Ds1307ControlRegAddress_U8  0x07u   // Address to access Ds1307 CONTROL register

#include "inc/tm4c123gh6pm.h"
#include <stdint.h>
#include <stdbool.h>
#include <./inc/tm4c123gh6pm.h>
#include <inc/hw_memmap.h>
#include <inc/hw_types.h>
#include <driverlib/gpio.h>
#include <driverlib/pin_map.h>
#include <driverlib/sysctl.h>
#include <driverlib/uart.h>
#define sw1 0x10
#define sw2 0x01
#define sw1_on 0x00
#define sw2_on 0x00
#define sw1_off 0x10
#define sw2_off 0x01
#define led_g 0x08
#define led_r 0x02
#define led_b 0x04
#define PWM_max 650
#define PWM_min 150
#define MAX_BUF_SIZE    56
#define SLAVE_ADDR      0x68             /* 1100 1000 */
#define I2C_START_ADDR  0x08
#define RS 6
#define EN 7

const char segment_code[] = { 0x3F, 0x06, 0x5B, 0x4F, 0x66, 0x6D, 0x7D, 0x07,
                              0x7F, 0x6F }; // 7 segment codes array

void delayMs(int n)
{
    int i, j;
    for(i =0; i < n; i++)
        for(j =0; j <600; j++){}/* do nothing for 1 ms */
}

void configure_gpio()
{
    SYSCTL_RCGC2_R |= 0x00000023;   /* enable clock to GPIOF at clock gating control register */
    GPIO_PORTF_LOCK_R = 0x4C4F434B; /* unlock commit register */
    GPIO_PORTA_LOCK_R = 0x4C4F434B; /* unlock commit register */
    GPIO_PORTB_LOCK_R = 0x4C4F434B; /* unlock commit register */
    GPIO_PORTF_CR_R = 0x1F;         /* make PORTF configurable */
    GPIO_PORTA_CR_R = 0xF0;         /* make PORTA configurable */
    GPIO_PORTB_CR_R = 0xFF;         /* make PORTB configurable */

    GPIO_PORTF_DIR_R = 0x0E;
    /* enable the GPIO pins for the LED (PF3, 2 1) as output */
    GPIO_PORTA_DIR_R = 0xF0;
    /* enable the GPIO pins for the segment as output */
    GPIO_PORTB_DIR_R = 0xFF;
    /* enable the GPIO pins for the segment as output */
    GPIO_PORTF_DEN_R = 0x1F;
    /* enable the GPIO pins for digital function */
    GPIO_PORTA_DEN_R = 0xF0;
    /* enable the GPIO pins for digital function */
    GPIO_PORTB_DEN_R = 0xFF;
    /* enable the GPIO pins for digital function */

    GPIO_PORTF_PUR_R = 0x11;
    /* enable pull up for pin 4 ,0 */

    GPIO_PORTF_IS_R  = 0x00;
    GPIO_PORTF_IBE_R = 0x00;
    GPIO_PORTF_IEV_R = 0x00;
    GPIO_PORTF_ICR_R = 0xff;
    //enable interrupt on sw1 and sw2

    GPIO_PORTF_IM_R = 0x11;
    NVIC_EN0_R |= (1<<30);
}
void configure_systick()
{
    NVIC_ST_CTRL_R      = 0;            // Disable timer during setup
    NVIC_ST_RELOAD_R    = 16000-1;      // This should expire once every 1ms
    NVIC_ST_CTRL_R      = 0x00000007;   // Clock src set to system clock and enable timer and interrupt
}

void led_set(char led_mask, char set_reset)
{
    GPIO_PORTF_DATA_R = (set_reset == 0)?(GPIO_PORTF_DATA_R&(~led_mask)):(GPIO_PORTF_DATA_R|led_mask);
    // One-liner for if else and bit masking! :P
}
void led_block_set(char led_mask, char set_reset)
{
    GPIO_PORTF_DATA_R = (set_reset == 0)?0:led_mask; // Over-writes previous LEDs!!
}
void display_digit_dp(int digit)
{
    // Uses segment code look up table to output a 7 segment digit
    // code on PORTB. If 7th bit of input digit is set dot pointer
    // will also be lit. e.g. display_digit_dp(5+0x80) will display
    // 5.
    GPIO_PORTB_DATA_R = segment_code[(digit&0x0F)]+(digit&0x80);

}
/* global enable interrupts */
void configure_uart()
{
        SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);
        SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

        GPIOPinConfigure(GPIO_PA0_U0RX);
        GPIOPinConfigure(GPIO_PA1_U0TX);
        GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

        UARTConfigSetExpClk(UART0_BASE, SysCtlClockGet(), 115200,(UART_CONFIG_WLEN_8 | UART_CONFIG_STOP_ONE | UART_CONFIG_PAR_NONE));
        UART0_IM_R |= (1<<4)|(1<<6) ;
        UART0_IFLS_R = 0;
        NVIC_EN0_R |= (1<<5);
}

void configure_pwm1()
{
        SYSCTL_RCGCPWM_R    |= SYSCTL_RCGCPWM_R1;       /* Enable clock to PWM1 module */
        SYSCTL_RCGCGPIO_R   |= SYSCTL_RCGCGPIO_R3;      /* Enable system clock to PORTD */
        SYSCTL_RCC_R        |= (SYSCTL_RCC_PWMDIV_2|SYSCTL_RCC_USEPWMDIV);
        /* Feed clock to PWM1 module with divider of 2 */
        /* Setting of PD1 pin for M1PWM1 channel output pin */
        GPIO_PORTD_AFSEL_R |= (1<<1);       /* PD1 sets a alternate function */
        GPIO_PORTD_PCTL_R  |= 0x00000050;   /* make PD1 PWM output pin       */
        GPIO_PORTD_DEN_R   |= (1<<1);       /* set PD1 as a digital pin      */
        /* PWM1 channel 0 setting */
        PWM1_0_CTL_R &= ~(1<<0);      /* Disable Generator 3 counter */
        PWM1_0_CTL_R &= ~(1<<1);      /* select down count mode of counter 3*/
        PWM1_0_GENB_R = 0x0000080C;   /* Set PWM output when counter reloaded and clear when matches PWMCMPA */
        PWM1_0_LOAD_R = 5000;         /* set load value for 1kHz (16MHz/16000) */
        //PWM1_3_CMPA_R = 4320;       /* set duty cycle to 50% by loading of 16000 to PWM1CMPA */
        PWM1_0_CTL_R = 1;             /* Enable Generator 0 counter */
        PWM1_ENABLE_R = (1<<1);       /* Enable PWM1 channel 1 output */
}
void configure_timer_1(void)

{
    SYSCTL_RCGCTIMER_R |= (1<<1);           /*enable clock Timer1 sub-timer A in run mode       */
    TIMER1_CTL_R        = 0;                /* disable timer1 output                            */
    TIMER1_CFG_R        = 0x4;              /*select 16-bit configuration option                */
    TIMER1_TAMR_R       = ((1<<1)|(1<<10)|(1<<5)); /*select periodic down counter mode of timer1*/
    TIMER1_TAPR_R       = 255;              /* TimerA pre scaler value                          */
    TIMER1_TAILR_R      = 12800-1 ;         /* TimerA counter starting count down value         */
    TIMER1_ICR_R       |= 0x11;             /* TimerA timeout flag bit clears                   */
    TIMER1_IMR_R        =((1<<0)|(1<<4));   /*enables TimerA time-out interrupt mask            */
    TIMER1_CTL_R       |= 0x01;             /* Enable TimerA module                             */
    NVIC_EN0_R         |= (1<<21);          /*enable IRQ21                                      */
}

void configure_adc()
{
        /* enable clocks */
        SYSCTL_RCGCGPIO_R |= 0x10; /* enable clock to PE (AIN0 is on PE3)               */
        SYSCTL_RCGCADC_R |= 1;     /* enable clock to ADC0                              */
                                   /* initialize PE3 for AIN0 input                     */
        GPIO_PORTE_AFSEL_R |= 8;   /* enable alternate function                         */
        GPIO_PORTE_DEN_R &= ~8;    /* disable digital function                          */
        GPIO_PORTE_AMSEL_R |= 8;   /* enable analog function                            */
                                   /* initialize ADC0                                   */
        ADC0_ACTSS_R &= ~8;        /* disable SS3 during configuration                  */
        ADC0_EMUX_R &= ~0xF000;    /* software trigger conversion                       */
        ADC0_SSMUX3_R = 0;         /* get input from channel 0                          */
        ADC0_SSCTL3_R |= 6;        /* take one sample at a time, set flag at 1st sample */
        ADC0_ACTSS_R |= 8;         /* enable ADC0 sequencer 3                           */
}

void configure_I2C2(void)
{

    SYSCTL_RCGCI2C_R    |= 0x04;                /* enable clock to I2C2         */
    SYSCTL_RCGCGPIO_R   |= (1<<4);              /* enable clock to GPIOE        */
                                                /* PORTE 4, 5 for I2C2          */
    GPIO_PORTE_AFSEL_R  |= ((1<<4)|(1<<5));     /* PORTE 4, 5 for I2C2          */
    GPIO_PORTE_PCTL_R   |= ((3<<20)|(3<<16));   /* PORTE 4, 5 for I2C2          */
    GPIO_PORTE_DEN_R    |= 0x30;                /* PORTE 4, 5 as digital pins   */
    GPIO_PORTE_ODR_R    |= 0x20;                /* PORTE 5 as open drain        */
    I2C2_MCR_R = 0x10;                          /* master mode                  */
    I2C2_MTPR_R = 7;                            /* 100 kHz @ 16 MHz             */

}
/* Wait until I2C master is not busy and return error code */
/* If there is no error, return 0 */
static int I2C2_wait_till_done(void)
{
    while(I2C2_MCS_R & 1);        /* wait until I2C master is not busy  */
    return I2C2_MCS_R & 0xE;      /* return I2C error code              */
}

/* Write one byte only */
/* byte write: S-(saddr+w)-ACK-maddr-ACK-data-ACK-P */
char I2C2_byteWrite(int slaveAddr, char memAddr, char data)
{
    char error;
    /* send slave address and starting address */
    I2C2_MSA_R = slaveAddr << 1;
    I2C2_MDR_R = memAddr;
    I2C2_MCS_R = 3;                     /* S-(saddr+w)-ACK-maddr-ACK    */
    error = I2C2_wait_till_done();      /* wait until write is complete */
    if(error) return error;
    /* send data */
    I2C2_MDR_R = data;
    I2C2_MCS_R = 5;                     /* -data-ACK-P                  */
    error = I2C2_wait_till_done();      /* wait until write is complete */
    while( I2C2_MCS_R & 0x40 );         /* wait until bus is not busy   */
    error = I2C2_MCS_R & 0xE;
    if(error) return error;
    return 0;                           /* no error */
}

char I2C2_bytesWrite(int slaveAddr, char memAddr, char *data, char size)
{
    char error;
    int i;
                                        /* send slave address and starting address */
    I2C2_MSA_R = slaveAddr << 1;
    I2C2_MDR_R = memAddr;
    I2C2_MCS_R = 3;                     /* S-(saddr+w)-ACK-maddr-ACK    */
    error = I2C2_wait_till_done();      /* wait until write is complete */
    if(error) return error;
    /* send data */
    for(i = 0; i < size-1 ;i++)
    {
        I2C2_MDR_R = data[i];
        I2C2_MCS_R = 1;                 /* -data-ACK-P                  */
        error = I2C2_wait_till_done();  /* wait until write is complete */
        if (error) return error;
    }
    I2C2_MDR_R = data[i];
    I2C2_MCS_R = 5;                 /* -data-ACK-P                  */
    error = I2C2_wait_till_done();  /* wait until write is complete */
    while( I2C2_MCS_R & 0x40 );
    if(error) return error;
    return 0;                           /* no error */
}
/* Read memory */
/* read: S-(saddr+w)-ACK-maddr-ACK-R-(saddr+r)-ACK-data-ACK-data-ACK-...-data-NACK-P */


char I2C2_read(int slaveAddr, char memAddr, int byteCount, char* data)
{
    char error;
    if (byteCount <= 0)
        return -1;       /* no read was performed */
    /* send slave address and starting address */
    I2C2_MSA_R = slaveAddr << 1;
    I2C2_MDR_R = memAddr;
    I2C2_MCS_R = 3;                 /* S-(saddr+w)-ACK-maddr-ACK */
    error = I2C2_wait_till_done();
    if(error)
        return error;
    /* to change bus from write to read, send restart with slave addr */
    I2C2_MSA_R = (slaveAddr << 1) + 1;   /* restart: -R-(saddr+r)-ACK */
    if( byteCount == 1 )            /* if last byte, don't ack */
        I2C2_MCS_R = 7;             /* -data-NACK-P */
    else                            /* else ack */
        I2C2_MCS_R = 0xB;           /* -data-ACK- */
    error = I2C2_wait_till_done();
    if(error) return error;
    *data++ = I2C2_MDR_R;           /* store the data received */
    if( --byteCount == 0 ) {        /* if single byte read, done */

        while( I2C2_MCS_R & 0x40 ); /* wait until bus is not busy */
        return 0;                   /* no error */
    }

    /* read the rest of the bytes */
    while( byteCount > 1 ) {
        I2C2_MCS_R = 9;             /* -data-ACK- */
        error = I2C2_wait_till_done();
        if(error) return error;
        byteCount--;
        *data++ = I2C2_MDR_R;       /* store data received */
    }
    I2C2_MCS_R = 5;                 /* -data-NACK-P */
    error = I2C2_wait_till_done();
    *data = I2C2_MDR_R;             /* store data received */
    while( I2C2_MCS_R & 0x40 );     /* wait until bus is not busy */
    return 0;                       /* no error */
}

void configure_spi()
{
    SYSCTL_RCGCSSI_R |= (1<<3);/* enable clock to SSI3 */
    SYSCTL_RCGCGPIO_R |= 8;/* enable clock to GPIOD for SSI1 */
    SYSCTL_RCGCGPIO_R |= 0x20;/* enable clock to GPIOF for slave select */
    /* configure PORTD 3, 1 for SSI1 clock and Tx */
    GPIO_PORTD_AMSEL_R &= ~0x09;/* disable analog for these pins */
    GPIO_PORTD_DEN_R |= 0x09;/* and make them digital */
    GPIO_PORTD_AFSEL_R |= 0x09;/* enable alternate function */
    GPIO_PORTD_PCTL_R &= ~0x0000F00F;/* assign pins to SSI3 */
    GPIO_PORTD_PCTL_R |= 0x00001001;/* assign pins to SSI3 */
    /* configure PORTF 2 for slave select */
    GPIO_PORTF_DEN_R |= 0x04;/* make the pin digital */
    GPIO_PORTF_DIR_R |= 0x04;/* make the pin output */
    GPIO_PORTF_DATA_R |= 0x04;/* keep SS idle high */
    /* SPI Master, POL = 0, PHA = 0, clock = 4 MHz, 16 bit data */
    SSI3_CR1_R = 0;/* disable SSI and make it master */
    SSI3_CC_R = 0;/* use system clock */
    SSI3_CPSR_R = 2;/* prescaler divided by 2 */
    SSI3_CR0_R = 0x0007;/* 8 MHz SSI clock, SPI mode, 8 bit data */
    SSI3_CR1_R |= 2;/* enable SSI3 */
}

void send_dac(unsigned int data, unsigned int command)
{
    int temp =  GPIO_PORTF_DATA_R ;
    unsigned temp_data = ((data<<2)|(command));
    GPIO_PORTF_DATA_R &= ~0x04; /* assert SS low */
    while((SSI3_SR_R & 2) == 0);/* wait until FIFO not full */
    SSI3_DR_R = (temp_data>>8);
    while(SSI3_SR_R & 0x10);    /* wait until transmit complete */
    SSI3_DR_R = temp_data;           /* transmit high byte */
    while(SSI3_SR_R & 0x10);    /* wait until transmit complete */
    GPIO_PORTF_DATA_R |= 0x04;
    GPIO_PORTF_DATA_R = temp;  /* keep SS idle high */
}

void send_command(int command)
{
    GPIO_PORTA_DATA_R |= (1<<EN);
    GPIO_PORTA_DATA_R &= ~(1<<RS);
    delayMs(5);
    GPIO_PORTB_DATA_R = command;
    delayMs(5);
    GPIO_PORTA_DATA_R &= ~(1<<EN);
    delayMs(5);
}

void send_command_f(int command)
{
    GPIO_PORTA_DATA_R |= (1<<EN);
    GPIO_PORTA_DATA_R &= ~(1<<RS);
    delayMs(1);
    GPIO_PORTB_DATA_R = command;
    delayMs(1);
    GPIO_PORTA_DATA_R &= ~(1<<EN);
    delayMs(1);
}

void resetcursor()
{
    send_command_f(1<<1);
}

void clear_lcd()
{
    send_command_f((1<<0));       // Clear Display
}

void init_lcd()
{
    send_command((1<<5)+(1<<4));
    send_command((1<<5)+(1<<4));
    send_command((1<<5)+(1<<4)+(1<<3));  //Set data lines and font size
    send_command((1<<3));       // Off display
    send_command((1<<0));       // Clear Display
    send_command((1<<2)+(1<<1));// Entry Mode Set
}

void send_data(char ch)
{
    GPIO_PORTA_DATA_R |= (1<<EN);
    GPIO_PORTA_DATA_R |= (1<<RS);
    delayMs(1);
    GPIO_PORTB_DATA_R = ch;
    delayMs(1);
    GPIO_PORTA_DATA_R &= ~(1<<EN);
    delayMs(1);
}


#endif /* PERIPH_CTL_H_ */
