/* Port Mapping
 *
 * 1. Motor for lifting car (PD 3 pins)
 * 2. Motor for door (PD 3 pins)
 * 3. Ultrasonic sensor for floor detection (PF)
 * 4. Ultrasonic sensor for door obstacle (PF)
 * 5. 4x4 keypad for inputs (PE-3210, PC-7654)
 * 6. LCD to display lift status (PA, PB)
 */

/* Keypad Mapping
 *
 *  0 (Floor 2)  |  1 (Open)   |   2 ()              |    3 (Floor 2 Down)
 *  4 (Floor 1)  |  5 ()       |   6 (Floor 1 Down)  |    7 (Floor 1 Up)
 *  8 (Floor 0)  |  9 ()       |  10 ()              |   11 (Floor 0 Up)
 * 12 ()         | 13 ()       |  14 ()              |   15 ()
 */

#include <./inc/tm4c123gh6pm.h>
#include <stdlib.h>
#include <stdio.h>
#include <uart_logger.h>
#include <lcd_status_viewer.h>
#include <systick_configurator.h>
#include <keypad_manager.h>
#include <ultrasonic_distance_calculator.h>
#include <dc_motor_controller.h>

#define HIGH 0xFFFF
#define LOW 0x0000
#define _0_MILLI_S 0
#define _10_MILLI_S 10
#define _20_MILLI_S 20
#define _30_MILLI_S 30
#define _200_MILLI_S 2000
#define SAMPLE_ZERO 0
#define SAMPLE_ONE 1
#define SAMPLE_TWO 2
#define SAMPLE_THREE 3
#define NO_KEY_PRESSED 20
#define ELEVATOR_STOPPED_DOOR_CLOSED 0
#define ELEVATOR_STOPPED_DOOR_OPENING 1
#define ELEVATOR_STOPPED_DOOR_OPEN 2
#define ELEVATOR_STOPPED_DOOR_CLOSING 3
#define ELEVATOR_ASCEND 4
#define ELEVATOR_DESCEND 4
#define FLOOR_2_KEY 0
#define FLOOR_1_KEY 4
#define FLOOR_0_KEY 8
#define OPEN_KEY 1
#define CLOSE_KEY 5
#define FLOOR_1_DOWN_KEY 6
#define FLOOR_2_DOWN_KEY 3
#define FLOOR_1_UP_KEY 7
#define FLOOR_0_UP_KEY 11
#define FLOOR_2 2
#define FLOOR_1 1
#define FLOOR_0 0
#define DOOR_OPEN 1
#define DOOR_CLOSE 0

void enableInterruptsGlobally();
void displayCurrentInputs();
void handleKeypadInput();
void measureCarHeight();
void handleElevator();
void handleCurrentStateClosed();
void controlDoor(int open_or_close);
void operateElevator();

unsigned int start_keypad_debouncer = FALSE, keypad_debounce_time = _0_MILLI_S,
    keypad_press_complete = FALSE, keypad_key_pressed = NO_KEY_PRESSED,
    keypad_previous_sample, delay_for_ultrasonic = _0_MILLI_S,
    ready_to_measure_height = TRUE, current_elevator_state = ELEVATOR_STOPPED_DOOR_CLOSED,
    next_elevator_state = ELEVATOR_STOPPED_DOOR_CLOSED, current_floor_level = FLOOR_0,
    previous_floor_level = FLOOR_0, next_floor_level = FLOOR_0;
unsigned int car_height;
unsigned short int keypad_input[16] = {LOW, LOW, LOW, LOW, LOW, LOW, LOW, LOW,
                                       LOW, LOW, LOW, LOW, LOW, LOW, LOW, LOW};


int main() {
    enableInterruptsGlobally();
    systickSetup();
    UARTInit();
    LCDInit();
    keyPadPortsCEInit();
    timer0B1AAndPortFInit();
    PWMForMotorInit();
    LCDPrintString("Floor 0");

    while (1) {
        handleKeypadInput();
        operateElevator();
    }
}

void controlDoor(int open_or_close) {
    if (open_or_close == DOOR_CLOSE) {
        for (int k = 0; k < 800; k = k+50) {
            GPIO_PORTE_DATA_R = 0x0E;
            if ((GPIO_PORTC_DATA_R & 0xF0) == 0xD0) {
                keypad_input[OPEN_KEY] = LOW;
                for ( ; k >= 0; k = k-50) {
                    for(int i=0; i<50; i++) {
                        GPIO_PORTD_DATA_R |= (1<<1);
                        timer1AMicroSecondDelay(1000 + k);
                        GPIO_PORTD_DATA_R &= ~(1<<1);
                        timer1AMicroSecondDelay(2000 - k);
                    }
                }
                timer1AMicroSecondDelay(200000);
                k = 0;
            }
            for(int i=0; i<50; i++) {
                GPIO_PORTD_DATA_R |= (1<<1);
                timer1AMicroSecondDelay(1000 + k);
                GPIO_PORTD_DATA_R &= ~(1<<1);
                timer1AMicroSecondDelay(2000 - k);
            }
            GPIO_PORTE_DATA_R = 0x00;
        }
    } else {
        for (int k = 800; k >= 0; k = k-50) {
            for(int i=0; i<50; i++) {
                GPIO_PORTD_DATA_R |= (1<<1);
                timer1AMicroSecondDelay(1000 + k);
                GPIO_PORTD_DATA_R &= ~(1<<1);
                timer1AMicroSecondDelay(2000 - k);
            }
        }
    }
}


void operateElevator() {
    if (current_floor_level == FLOOR_0) {
        if (keypad_input[FLOOR_0_UP_KEY] == HIGH) {
            keypad_input[FLOOR_0_KEY] = LOW;
            controlDoor(DOOR_OPEN);
            keypad_input[FLOOR_0_UP_KEY] = LOW;
            timer1AMicroSecondDelay(200000);
            controlDoor(DOOR_CLOSE);
        } else if ((keypad_input[FLOOR_1_KEY] == HIGH) || (keypad_input[FLOOR_1_UP_KEY] == HIGH)) {
            LCDPrintString("Going Up  ");
            setMainDCMotorPulseAndDirection(6000, 0);
            timer1AMicroSecondDelay(600000);
            setMainDCMotorPulseAndDirection(15999, 0);
            LCDPrintString("Floor 1   ");
            timer1AMicroSecondDelay(1000000);
            keypad_input[FLOOR_1_KEY] = LOW;
            keypad_input[FLOOR_1_UP_KEY] = LOW;
            controlDoor(DOOR_OPEN);
            timer1AMicroSecondDelay(200000);
            controlDoor(DOOR_CLOSE);
            current_floor_level = FLOOR_1;
            previous_floor_level = FLOOR_0;
        } else if ((keypad_input[FLOOR_2_KEY] == HIGH) || (keypad_input[FLOOR_2_DOWN_KEY] == HIGH)) {
            LCDPrintString("Going Up  ");
            setMainDCMotorPulseAndDirection(6000, 0);
            timer1AMicroSecondDelay(1400000);
            setMainDCMotorPulseAndDirection(15999, 0);
            LCDPrintString("Floor 2   ");
            timer1AMicroSecondDelay(1000000);
            keypad_input[FLOOR_2_KEY] = LOW;
            keypad_input[FLOOR_2_DOWN_KEY] = LOW;
            controlDoor(DOOR_OPEN);
            timer1AMicroSecondDelay(200000);
            controlDoor(DOOR_CLOSE);
            current_floor_level = FLOOR_2;
            previous_floor_level = FLOOR_0;
        } else if (keypad_input[FLOOR_1_DOWN_KEY] == HIGH) {
            LCDPrintString("Going Up  ");
            setMainDCMotorPulseAndDirection(6000, 0);
            timer1AMicroSecondDelay(600000);
            setMainDCMotorPulseAndDirection(15999, 0);
            LCDPrintString("Floor 1   ");
            timer1AMicroSecondDelay(1000000);
            keypad_input[FLOOR_1_DOWN_KEY] = LOW;
            controlDoor(DOOR_OPEN);
            timer1AMicroSecondDelay(200000);
            controlDoor(DOOR_CLOSE);
            current_floor_level = FLOOR_1;
            previous_floor_level = FLOOR_0;
        }
    }
    if (current_floor_level == FLOOR_1) {
        keypad_input[FLOOR_1_KEY] = LOW;
        if (keypad_input[FLOOR_1_UP_KEY] == HIGH) {
            controlDoor(DOOR_OPEN);
            keypad_input[FLOOR_1_UP_KEY] = LOW;
            timer1AMicroSecondDelay(200000);
            controlDoor(DOOR_CLOSE);
        } else if (keypad_input[FLOOR_1_DOWN_KEY] == HIGH) {
            controlDoor(DOOR_OPEN);
            keypad_input[FLOOR_1_DOWN_KEY] = LOW;
            timer1AMicroSecondDelay(200000);
            controlDoor(DOOR_CLOSE);
        } else if ((keypad_input[FLOOR_2_KEY] == HIGH) || (keypad_input[FLOOR_2_DOWN_KEY] == HIGH)) {
            LCDPrintString("Going Up  ");
            setMainDCMotorPulseAndDirection(6000, 0);
            timer1AMicroSecondDelay(600000);
            setMainDCMotorPulseAndDirection(15999, 0);
            LCDPrintString("Floor 2   ");
            timer1AMicroSecondDelay(1000000);
            keypad_input[FLOOR_2_KEY] = LOW;
            keypad_input[FLOOR_2_DOWN_KEY] = LOW;
            controlDoor(DOOR_OPEN);
            timer1AMicroSecondDelay(200000);
            controlDoor(DOOR_CLOSE);
            current_floor_level = FLOOR_2;
            previous_floor_level = FLOOR_1;
        } else if (keypad_input[FLOOR_0_KEY] == HIGH || (keypad_input[FLOOR_0_UP_KEY] == HIGH)) {
            LCDPrintString("Going Down");
            setMainDCMotorPulseAndDirection(6000, 1);
            timer1AMicroSecondDelay(600000);
            setMainDCMotorPulseAndDirection(15999, 1);
            LCDPrintString("Floor 0   ");
            timer1AMicroSecondDelay(1000000);
            keypad_input[FLOOR_0_KEY] = LOW;
            keypad_input[FLOOR_0_UP_KEY] = LOW;
            controlDoor(DOOR_OPEN);
            timer1AMicroSecondDelay(200000);
            controlDoor(DOOR_CLOSE);
            current_floor_level = FLOOR_0;
            previous_floor_level = FLOOR_1;
        }
    }
    if (current_floor_level == FLOOR_2) {
        keypad_input[FLOOR_2_KEY] = LOW;
        if (keypad_input[FLOOR_2_DOWN_KEY] == HIGH) {
            controlDoor(DOOR_OPEN);
            keypad_input[FLOOR_2_DOWN_KEY] = LOW;
            timer1AMicroSecondDelay(2000000);
            controlDoor(DOOR_CLOSE);
        } else if ((keypad_input[FLOOR_1_KEY] == HIGH) || (keypad_input[FLOOR_1_DOWN_KEY] == HIGH)) {
            LCDPrintString("Going Down");
            setMainDCMotorPulseAndDirection(6000, 1);
            timer1AMicroSecondDelay(600000);
            setMainDCMotorPulseAndDirection(15999, 1);
            LCDPrintString("Floor 1   ");
            timer1AMicroSecondDelay(1000000);
            keypad_input[FLOOR_1_KEY] = LOW;
            keypad_input[FLOOR_1_DOWN_KEY] = LOW;
            controlDoor(DOOR_OPEN);
            timer1AMicroSecondDelay(200000);
            controlDoor(DOOR_CLOSE);
            current_floor_level = FLOOR_1;
            previous_floor_level = FLOOR_2;
        } else if ((keypad_input[FLOOR_0_KEY] == HIGH) || (keypad_input[FLOOR_0_UP_KEY] == HIGH)) {
            LCDPrintString("Going Down");
            setMainDCMotorPulseAndDirection(6000, 1);
            timer1AMicroSecondDelay(1600000);
            setMainDCMotorPulseAndDirection(15999, 1);
            LCDPrintString("Floor 0   ");
            timer1AMicroSecondDelay(1000000);
            keypad_input[FLOOR_0_KEY] = LOW;
            keypad_input[FLOOR_0_UP_KEY] = LOW;
            controlDoor(DOOR_OPEN);
            timer1AMicroSecondDelay(200000);
            controlDoor(DOOR_CLOSE);
            current_floor_level = FLOOR_0;
            previous_floor_level = FLOOR_1;
        } else if (keypad_input[FLOOR_1_UP_KEY] == HIGH) {
            LCDPrintString("Going Down");
            setMainDCMotorPulseAndDirection(6000, 1);
            timer1AMicroSecondDelay(600000);
            setMainDCMotorPulseAndDirection(15999, 1);
            LCDPrintString("Floor 1   ");
            timer1AMicroSecondDelay(1000000);
            keypad_input[FLOOR_1_UP_KEY] = LOW;
            controlDoor(DOOR_OPEN);
            timer1AMicroSecondDelay(200000);
            controlDoor(DOOR_CLOSE);
            current_floor_level = FLOOR_1;
            previous_floor_level = FLOOR_2;
        }
    }
}

void handleKeypadInput() {
    if (keypad_press_complete == TRUE) {
        keypad_press_complete = FALSE;
        keypad_input[keypad_key_pressed] = ~(keypad_input[keypad_key_pressed]);
        displayCurrentInputs();
    }
}

void displayCurrentInputs() {
    UARTPrint("\n\r___________________\n\r");
    UARTPrint("\n\rElevator Car Inputs\n\r");
    UARTPrint("Floor 2 :");
    if(keypad_input[0] == HIGH) {
        UARTPrint(" 1\n\r");
    } else {
        UARTPrint(" 0\n\r");
    }

    UARTPrint("Floor 1 :");
    if(keypad_input[4] == HIGH) {
        UARTPrint(" 1\n\r");
    } else {
        UARTPrint(" 0\n\r");
    }

    UARTPrint("Floor 0 :");
    if(keypad_input[8] == HIGH) {
        UARTPrint(" 1\n\r");
    } else {
        UARTPrint(" 0\n\r");
    }

    UARTPrint("Open\n\r");

    UARTPrint("\n\rFloor Inputs\n\r");
    UARTPrint("Floor 2 Down :");
    if(keypad_input[3] == HIGH) {
        UARTPrint(" 1\n\r");
    } else {
        UARTPrint(" 0\n\r");
    }

    UARTPrint("Floor 1 Up   :");
    if(keypad_input[7] == HIGH) {
        UARTPrint(" 1\n\r");
    } else {
        UARTPrint(" 0\n\r");
    }

    UARTPrint("Floor 1 Down :");
    if(keypad_input[6] == HIGH) {
        UARTPrint(" 1\n\r");
    } else {
        UARTPrint(" 0\n\r");
    }

    UARTPrint("Floor 0 Up   :");
    if(keypad_input[11] == HIGH) {
        UARTPrint(" 1\n\r");
    } else {
        UARTPrint(" 0\n\r");
    }
    UARTPrint("\n\r___________________\n\r");
}

void inline enableInterruptsGlobally() {
    __asm  ("CPSIE  I\n");
}

void systickHandler() {
    if (start_keypad_debouncer == TRUE) {
        if (keypad_debounce_time == _0_MILLI_S) {
            keypad_previous_sample = isKeyPressed();
            keypad_key_pressed = getPressedkey();
        }
        keypad_debounce_time++;
        if (keypad_debounce_time == _10_MILLI_S) {
            if (keypad_previous_sample != isKeyPressed()) {
                keypad_debounce_time = _0_MILLI_S;
            }
        }
        if (keypad_debounce_time == _20_MILLI_S) {
            if (keypad_previous_sample != isKeyPressed()) {
                keypad_debounce_time = _0_MILLI_S;
            }
        }
        if (keypad_debounce_time == _30_MILLI_S) {
            if (keypad_previous_sample != isKeyPressed()) {
                keypad_debounce_time = _0_MILLI_S;
            } else {
                keypad_press_complete = isKeyPressed();
                start_keypad_debouncer = FALSE;
                keypad_debounce_time = _0_MILLI_S;
            }
        }
    }
}

void GPIOPortCHandler() {
    start_keypad_debouncer = TRUE;
    GPIO_PORTC_ICR_R = CLEAR_PC_7_6_5_4_INTERRUPTS;
}
