/*
 * ultrasonic_sensor.c
 *
 *  Created on: 04-May-2024
 *      Author: krish
 */
#include <stdint.h>
#include <math.h>
#include <./inc/tm4c123gh6pm.h>

#include "UART_init.h"
#include "GPIO.h"
#include "ADC_PWM.h"
#include "core_logic.h"

/* This function captures consecutive rising and falling edges of a periodic signal */
/* from Timer Block 0 Timer A and returns the time difference (the period of the signal). */
uint32_t Measure_distance_from_front_sensor(void)
{
    int lastEdge, thisEdge;

      /* Given 10us trigger pulse */
      GPIO_PORTA_DATA_R &= ~(1<<4); /* make trigger  pin high */
      Delay_MicroSecond(10); /*10 seconds delay */
      GPIO_PORTA_DATA_R |= (1<<4); /* make trigger  pin high */
      Delay_MicroSecond(10); /*10 seconds delay */
      GPIO_PORTA_DATA_R &= ~(1<<4); /* make trigger  pin low */

    while(1)
    {
    TIMER0_ICR_R = 4;            /* clear timer0A capture flag */
    while((TIMER0_RIS_R & 4) == 0) ;    /* wait till captured */
    if(GPIO_PORTB_DATA_R & (1<<6)) /*check if rising edge occurs */
        {
            lastEdge = TIMER0_TAR_R;     /* save the timestamp */
            /* detect falling edge */
            TIMER0_ICR_R = 4;          /* clear timer0A capture flag */
            while((TIMER0_RIS_R & 4) == 0) ;    /* wait till captured */
            thisEdge = TIMER0_TAR_R;     /* save the timestamp */
            return (thisEdge - lastEdge); /* return the time difference */
        }
    }
}

/* This function captures consecutive rising and falling edges of a periodic signal */
/* from Timer Block 2 Timer A and returns the time difference (the period of the signal). */
uint32_t Measure_distance_from_left_sensor(void)
{
    int lastEdge, thisEdge;

      /* Given 10us trigger pulse */
      GPIO_PORTB_DATA_R &= ~(1<<4); /* make trigger  pin high */
      Delay_MicroSecond(10); /*10 seconds delay */
      GPIO_PORTB_DATA_R |= (1<<4); /* make trigger  pin high */
      Delay_MicroSecond(10); /*10 seconds delay */
      GPIO_PORTB_DATA_R &= ~(1<<4); /* make trigger  pin low */

    while(1)
    {
    TIMER1_ICR_R = 4;            /* clear timer2A capture flag */
    while((TIMER1_RIS_R & 4) == 0) ;    /* wait till captured */

    if(GPIO_PORTF_DATA_R & (1 << 2)) /*check if rising edge occurs */
        {
            lastEdge = TIMER1_TAR_R;     /* save the timestamp */
            /* detect falling edge */
            TIMER1_ICR_R = 4;          /* clear timer2A capture flag */
            while((TIMER1_RIS_R & 4) == 0) ;    /* wait till captured */
            thisEdge = TIMER1_TAR_R;     /* save the timestamp */
            return (thisEdge - lastEdge); /* return the time difference */
        }
    }
}

/* This function captures consecutive rising and falling edges of a periodic signal */
/* from Timer Block 3 Timer A and returns the time difference (the period of the signal). */
uint32_t Measure_distance_from_right_sensor(void)
{
    int lastEdge, thisEdge;

      /* Given 10us trigger pulse */
      GPIO_PORTB_DATA_R &= ~(1 << 3); /* make trigger  pin high */
      Delay_MicroSecond(10); /*10 seconds delay */
      GPIO_PORTB_DATA_R |= (1 << 3); /* make trigger  pin high */
      Delay_MicroSecond(10); /*10 seconds delay */
      GPIO_PORTB_DATA_R &= ~(1 << 3); /* make trigger  pin low */

    while(1)
    {
    TIMER3_ICR_R = 4;            /* clear timer3A capture flag */
    while((TIMER3_RIS_R & 4) == 0) ;    /* wait till captured */
    if(GPIO_PORTB_DATA_R & (1 << 2)) /*check if rising edge occurs */
        {
            lastEdge = TIMER3_TAR_R;     /* save the timestamp */
            /* detect falling edge */
            TIMER3_ICR_R = 4;          /* clear timer3A capture flag */
            while((TIMER3_RIS_R & 4) == 0) ;    /* wait till captured */
            thisEdge = TIMER3_TAR_R;     /* save the timestamp */
            return (thisEdge - lastEdge); /* return the time difference */
        }
    }
}


/* Timer0A initialization function */
/* Initialize Timer0A in input-edge time mode with up-count mode */
void Timer0ACapture_init(void)
{
    SYSCTL_RCGCTIMER_R |= 1;     /* enable clock to Timer Block 0 */
    SYSCTL_RCGCGPIO_R |= 2;      /* enable clock to PORTB */

    GPIO_PORTB_DIR_R &= ~0x40;        /* make PB6 an input pin */
    GPIO_PORTB_DEN_R |= 0x40;         /* make PB6 as digital pin */
    GPIO_PORTB_AFSEL_R |= 0x40;       /* use PB6 alternate function */
    GPIO_PORTB_PCTL_R &= ~0x0F000000;  /* configure PB6 for T0CCP0 */
    GPIO_PORTB_PCTL_R |= 0x07000000;

    /* PA4 as a digital output signal to provide trigger signal */
    SYSCTL_RCGCGPIO_R |= 1;      /* enable clock to PORTA */
    GPIO_PORTA_DIR_R |=(1<<4);         /* set PA4 as a digital output pin */
    GPIO_PORTA_DEN_R |=(1<<4);         /* make PA4 as digital pin */

    TIMER0_CTL_R &= ~1;          /* disable timer0A during setup */
    TIMER0_CFG_R = 4;            /* 16-bit timer mode */
    TIMER0_TAMR_R = 0x17;        /* up-count, edge-time, capture mode */
    TIMER0_CTL_R |=0x0C;        /* capture the rising and falling edge */
    TIMER0_CTL_R |= (1<<0);           /* enable timer0A */

}

/* Time0B initialization function */
/* Initialize Timer 2A in input-edge time mode with up-count mode */
void Timer1ACapture_init(void)
{
    SYSCTL_RCGCTIMER_R |= 2;     /* enable clock to Timer Block 2 */
    SYSCTL_RCGCGPIO_R |= 0x20;      /* enable clock to PORTF */
    SYSCTL_RCGCGPIO_R |= 0x2;      /* enable clock to PORTB */

    GPIO_PORTF_DIR_R &= ~0x04;        /* make PF2 an input pin */
    GPIO_PORTF_DEN_R |= 0x04;         /* make PF2 as digital pin */
    GPIO_PORTF_AFSEL_R |= 0x04;       /* use PF2 alternate function */
    GPIO_PORTF_PCTL_R &= ~0x00000F00;  /* configure PF2 for T2CCP0 */
    GPIO_PORTF_PCTL_R |= 0x00000700;

    /* PB4 as a digital output signal to provide trigger signal */
    SYSCTL_RCGCGPIO_R |= 2;      /* enable clock to PORTB */
    GPIO_PORTB_DIR_R |= (1<<4);         /* set PB4 as a digital output pin */
    GPIO_PORTB_DEN_R |= (1<<4);         /* make PB4 as digital pin */

    TIMER1_CTL_R &= ~0x1;          /* disable timerA during setup */
    TIMER1_CFG_R = 0x4;            /* 16-bit timer mode */
    TIMER1_TAMR_R = 0x17;        /* up-count, edge-time, capture mode */
    TIMER1_CTL_R |= 0xC;        /* capture the rising and falling edge */
    TIMER1_CTL_R |= (1 << 0);           /* enable timer2A */
}


/* Timer3A initialization function */
/* Initialize Timer3A in input-edge time mode with up-count mode */
void Timer3ACapture_init(void)
{
    SYSCTL_RCGCTIMER_R |= 8;     /* enable clock to Timer Block 3 */
    SYSCTL_RCGCGPIO_R |= 2;      /* enable clock to PORTB */

    GPIO_PORTB_DIR_R &= ~0x04;        /* make PB2 an input pin */
    GPIO_PORTB_DEN_R |= 0x04;         /* make PB2 as digital pin */
    GPIO_PORTB_AFSEL_R |= 0x04;       /* use PB2 alternate function */
    GPIO_PORTB_PCTL_R &= ~0x00000F00;  /* configure PB2 for T0CCP0 */
    GPIO_PORTB_PCTL_R |= 0x00000700;

    /* PA4 as a digital output signal to provide trigger signal */
    SYSCTL_RCGCGPIO_R |= 2;      /* enable clock to PORTB */
    GPIO_PORTB_DIR_R |= (1 << 3);         /* set PB3 as a digital output pin */
    GPIO_PORTB_DEN_R |= (1 << 3);         /* make PB3 as digital pin */

    TIMER3_CTL_R &= ~1;          /* disable timer3A during setup */
    TIMER3_CFG_R = 4;            /* 16-bit timer mode */
    TIMER3_TAMR_R = 0x17;        /* up-count, edge-time, capture mode */
    TIMER3_CTL_R |=0x0C;        /* capture the rising and falling edge */
    TIMER3_CTL_R |= (1<<0);           /* enable timer3A */
}

/* Create one microsecond second delay using Timer block 1 and sub timer A */

void Delay_MicroSecond(int time)
{
    int i;
    SYSCTL_RCGCTIMER_R |= 4;     /* enable clock to Timer Block 1 */
    TIMER2_CTL_R = 0;            /* disable Timer before initialization */
    TIMER2_CFG_R = 0x04;         /* 16-bit option */
    TIMER2_TAMR_R = 0x02;        /* periodic mode and down-counter */
    TIMER2_TAILR_R = 16 - 1;  /* TimerA interval load value reg */
    TIMER2_ICR_R = 0x1;          /* clear the TimerA timeout flag */
    TIMER2_CTL_R |= 0x01;        /* enable Timer A after initialization */

    for(i = 0; i < time; i++)
    {
        while ((TIMER2_RIS_R & 0x1) == 0) ;      /* wait for TimerA timeout flag */
        TIMER2_ICR_R = 0x1;      /* clear the TimerA timeout flag */
    }
}
