//*****************************************************************************
//
// compdcm_mpu9150.c - Example use of the SensorLib with the MPU9150
//
// Copyright (c) 2013-2017 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
//
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
//
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
//
// This is part of revision 2.1.4.178 of the EK-TM4C123GXL Firmware Package.
//
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_memmap.h"
//#include "inc/hw_ints.h"
#include "driverlib/debug.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/sysctl.h"
#include "driverlib/uart.h"
#include "utils/uartstdio.h"
#include "sensorlib/hw_mpu9150.h"
#include "sensorlib/hw_ak8975.h"
#include "sensorlib/i2cm_drv.h"
#include "sensorlib/ak8975.h"
#include "sensorlib/mpu9150.h"
#include "sensorlib/comp_dcm.h"
#include "rgb.h"
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "C:\Users\shish\Downloads\software\TivaWare_C_Series-2.2.0.295\TivaWare_C_Series-2.2.0.295\inc\tm4c123gh6pm.h"

#include "DTW.h"             // Library file for DTW algorithm

#include "Ultrasonic.h"
#include "Timer.h"
#include "UART.h"
#include "PWM.h"
#include "GPIO_init.h"
#include "Robo_init.h"         // Library file for initial testing

//*****************************************************************************
//
//! \addtogroup example_list
//! <h1>Nine Axis Sensor Fusion with the MPU9150 and Complimentary-Filtered
//! DCM (compdcm_mpu9150)</h1>
//!
//! This example demonstrates the basic use of the Sensor Library, TM4C123G
//! LaunchPad and SensHub BoosterPack to obtain nine axis motion measurements
//! from the MPU9150.  The example fuses the nine axis measurements into a set
//! of Euler angles: roll, pitch and yaw.  It also produces the rotation
//! quaternions.  The fusion mechanism demonstrated is complimentary-filtered
//! direct cosine matrix (DCM) algorithm is provided as part of the Sensor
//! Library.
//!
//! Connect a serial terminal program to the LaunchPad's ICDI virtual serial
//! port at 115,200 baud.  Use eight bits per byte, no parity and one stop bit.
//! The raw sensor measurements, Euler angles and quaternions are printed to
//! the terminal.  The RGB LED begins to blink at 1Hz after initialization is
//! completed and the example application is running.
//
//*****************************************************************************

//*****************************************************************************
//
// Define MPU9150 I2C Address.
//
//*****************************************************************************
#define MPU9150_I2C_ADDRESS     0x68

//*****************************************************************************
//
// Global array for holding the color values for the RGB.
//
//*****************************************************************************
uint32_t g_pui32Colors[3];

//*****************************************************************************
//
// Global instance structure for the I2C master driver.
//
//*****************************************************************************
tI2CMInstance g_sI2CInst;

//*****************************************************************************
//
// Global instance structure for the ISL29023 sensor driver.
//
//*****************************************************************************
tMPU9150 g_sMPU9150Inst;

//*****************************************************************************
//
// Global Instance structure to manage the DCM state.
//
//*****************************************************************************
tCompDCM g_sCompDCMInst;

//*****************************************************************************
//
// Global flags to alert main that MPU9150 I2C transaction is complete
//
//*****************************************************************************
volatile uint_fast8_t g_vui8I2CDoneFlag;

//*****************************************************************************
//
// Global flags to alert main that MPU9150 I2C transaction error has occurred.
//
//*****************************************************************************
volatile uint_fast8_t g_vui8ErrorFlag;

//*****************************************************************************
//
// Global flags to alert main that MPU9150 data is ready to be retrieved.
//
//*****************************************************************************
volatile uint_fast8_t g_vui8DataFlag;

//*****************************************************************************
//
// Global counter to control and slow down the rate of data to the terminal.
//
//*****************************************************************************
#define PRINT_SKIP_COUNT        10

uint32_t g_ui32PrintSkipCounter;

//*****************************************************************************
//
// The error routine that is called if the driver library encounters an error.
//
//*****************************************************************************
#ifdef DEBUG
void
__error__(char *pcFilename, uint32_t ui32Line)
{
}
#endif

//*****************************************************************************
//
// MPU9150 Sensor callback function.  Called at the end of MPU9150 sensor
// driver transactions. This is called from I2C interrupt context. Therefore,
// we just set a flag and let main do the bulk of the computations and display.
//
//*****************************************************************************
void MPU9150AppCallback(void *pvCallbackData, uint_fast8_t ui8Status)
{
    //
    // If the transaction succeeded set the data flag to indicate to
    // application that this transaction is complete and data may be ready.
    //
    if (ui8Status == I2CM_STATUS_SUCCESS)
    {
        g_vui8I2CDoneFlag = 1;
    }

    //
    // Store the most recent status in case it was an error condition
    //
    g_vui8ErrorFlag = ui8Status;
}

//*****************************************************************************
//
// Called by the NVIC as a result of GPIO port B interrupt event. For this
// application GPIO port B pin 2 is the interrupt line for the MPU9150
//
//*****************************************************************************
void IntGPIOb(void)
{
    unsigned long ulStatus;

    ulStatus = GPIOIntStatus(GPIO_PORTB_BASE, true);

    //
    // Clear all the pin interrupts that are set
    //
    GPIOIntClear(GPIO_PORTB_BASE, ulStatus);

    if (ulStatus & GPIO_PIN_2)
    {
        //
        // MPU9150 Data is ready for retrieval and processing.
        //
        MPU9150DataRead(&g_sMPU9150Inst, MPU9150AppCallback, &g_sMPU9150Inst);
    }
}

//*****************************************************************************
//
// Called by the NVIC as a result of I2C3 Interrupt. I2C3 is the I2C connection
// to the MPU9150.
//
//*****************************************************************************
void MPU9150I2CIntHandler(void)
{
    //
    // Pass through to the I2CM interrupt handler provided by sensor library.
    // This is required to be at application level so that I2CMIntHandler can
    // receive the instance structure pointer as an argument.
    //
    I2CMIntHandler(&g_sI2CInst);
}

//*****************************************************************************
//
// MPU9150 Application error handler. Show the user if we have encountered an
// I2C error.
//
//*****************************************************************************
void MPU9150AppErrorHandler(char *pcFilename, uint_fast32_t ui32Line)
{
    //
    // Set terminal color to red and print error status and locations
    //
    UARTprintf("\033[31;1m");
    UARTprintf("Error: %d, File: %s, Line: %d\n"
               "See I2C status definitions in sensorlib\\i2cm_drv.h\n",
               g_vui8ErrorFlag, pcFilename, ui32Line);

    //
    // Return terminal color to normal
    //
    UARTprintf("\033[0m");

    //
    // Set RGB Color to RED
    //
    g_pui32Colors[0] = 0xFFFF;
    g_pui32Colors[1] = 0;
    g_pui32Colors[2] = 0;
    RGBColorSet(g_pui32Colors);

    //
    // Increase blink rate to get attention
    //
    RGBBlinkRateSet(10.0f);

    //
    // Go to sleep wait for interventions.  A more robust application could
    // attempt corrective actions here.
    //
    while (1)
    {
        //
        // Do Nothing
        //
    }
}

//*****************************************************************************
//
// Function to wait for the MPU9150 transactions to complete. Use this to spin
// wait on the I2C bus.
//
//*****************************************************************************
void MPU9150AppI2CWait(char *pcFilename, uint_fast32_t ui32Line)
{
    //
    // Put the processor to sleep while we wait for the I2C driver to
    // indicate that the transaction is complete.
    //
    while ((g_vui8I2CDoneFlag == 0) && (g_vui8ErrorFlag == 0))
    {
        //
        // Do Nothing
        //
    }

    //
    // If an error occurred call the error handler immediately.
    //
    if (g_vui8ErrorFlag)
    {
        MPU9150AppErrorHandler(pcFilename, ui32Line);
    }

    //
    // clear the data flag for next use.
    //
    g_vui8I2CDoneFlag = 0;
}

//*****************************************************************************
//
// Configure the UART and its pins.  This must be called before UARTprintf().
//
//*****************************************************************************
void ConfigureUART(void)
{
    //
    // Enable the GPIO Peripheral used by the UART.
    //
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    //
    // Enable UART0
    //
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);

    //
    // Configure GPIO Pins for UART mode.
    //
    ROM_GPIOPinConfigure(GPIO_PA0_U0RX);
    ROM_GPIOPinConfigure(GPIO_PA1_U0TX);
    ROM_GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    //
    // Use the internal 16MHz oscillator as the UART clock source.
    //
    UARTClockSourceSet(UART0_BASE, UART_CLOCK_PIOSC);

    //
    // Initialize the UART for console I/O.
    //
    UARTStdioConfig(0, 115200, 16000000);
}

///////// Calibration values. Gesture dictionary ////////////////////////////

////////////////////           UP              //////////////////////////////////
const float X_up[] ={0.658,  0.256, -2.462, -1.959, -2.665, -1.819, -1.74,  -1.788, -1.696, -1.51};
const float Y_up[]  = {0.251, 0.063, 4.759, 5.886, 7.648, 7.778, 7.866, 8.243, 8.394, 7.979};
const float Z_up[] ={-9.658, -9.069, -9.484, -7.279, -5.883, -5.159, -4.698, -4.347, -5.924, -4.623};
//////////////////////        Down          /////////////////////////////////
const float X_down[] ={0.134, 0.079, 0.588, 0.526, 0.405, 0.04,  0.035, 0.11,  0.033, 0.203};
const float Y_down[] ={ 0.049,  0.099, -1.583, -4.084, -5.124, -5.0    ,-4.953, -4.983, -5.106, -4.922};
const float Z_down[] ={ -9.798,  -9.484, -10.224,  -8.868,  -8.571,  -8.3,    -8.318,  -8.366,  -8.507 ,-8.438};
//////////////////////        Left          /////////////////////////////////
const float X_left[] ={0.532,  -1.671,  -4.85,   -8.567, -11.295, -10.018, -10.36,  -10.098, -10.188,-10.018};
const float Y_left[] ={0.311, 0.38,  0.328, 0.617, 0.424, 0.729, 0.695, 0.511, 0.636, 0.605};
const float Z_left[] ={-9.656, -11.02,   -6.961,  -5.087,  -1.924,   0.185,   0.039,   0.288,   0.302, 0.075};
//////////////////////        Right          /////////////////////////////////
const float X_right[] ={0.527, 7.337, 8.103, 8.264, 8.581, 9.388, 9.385, 9.548, 9.726, 9.469};
const float Y_right[] ={0.423, 0.007, 0.128, 0.068, 0.09,  0.658, 1.136, 0.543, 0.803, 0.79};
const float Z_right[] ={-9.652, -3.864, -2.999, -1.112,  0.111,  0.486,  0.7,    0.371,  0.3,    0.397};
//////////////////////        Rest          /////////////////////////////////
const float X_rest[] ={0.122, 0.118, 0.076, 0.092, 0.14,  0.108, 0.098, 0.107, 0.105, 0.134};
const float Y_rest[] ={0.124, 0.125, 0.134, 0.086, 0.124, 0.135, 0.114, 0.122, 0.116, 0.14 };
const float Z_rest[] ={-9.593, -9.585, -9.603, -9.636, -9.609, -9.513, -9.634, -9.621, -9.568, -9.55 };

//*****************************************************************************
//
// Main application entry point.
//
//*****************************************************************************
int main(void)
{
    int_fast32_t i32IPart[16], i32FPart[16];
    uint_fast32_t ui32Idx, ui32CompDCMStarted;
    float pfData[16];
    float *pfAccel, *pfGyro, *pfMag, *pfEulers, *pfQuaternion;
//

    float buffer_A[3];              // Buffer to store data
    float X_present[10] = { 0 };    /// making it 50 or 20 causes a problem
    float Y_present[10] = { 0 };
    float Z_present[10] = { 0 };
    int k = 0;
    int w = 0;
  //////////// Variables to store values obtained from DTW algo //////////////
    float DTW_X_up, DTW_Y_up, DTW_Z_up, DTW_up_tot;
    float DTW_X_down, DTW_Y_down, DTW_Z_down, DTW_down_tot;
    float DTW_X_left, DTW_Y_left, DTW_Z_left, DTW_left_tot;
    float DTW_X_right, DTW_Y_right, DTW_Z_right, DTW_right_tot;
    float DTW_X_rest, DTW_Y_rest, DTW_Z_rest, DTW_rest_tot;

    float min;
    //
    // Initialize convenience pointers that clean up and clarify the code
    // meaning. We want all the data in a single contiguous array so that
    // we can make our pretty printing easier later.
    //
    pfAccel = pfData;
    pfGyro = pfData + 3;
    pfMag = pfData + 6;
    pfEulers = pfData + 9;
    pfQuaternion = pfData + 12;

    PWM_init();                        // PWM function
    PortE_Init();
    PortD_Init();
    PORTC_PWM_init();
    PWM1_1_CMPA_R = 1500;
    PWM0_3_CMPA_R = 2500;

    //
    // Setup the system clock to run at 40 Mhz from PLL with crystal reference
    //
    ROM_SysCtlClockSet(SYSCTL_SYSDIV_5 | SYSCTL_USE_PLL | SYSCTL_XTAL_16MHZ |
    SYSCTL_OSC_MAIN);

    //
    // Enable port B used for motion interrupt.
    //
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);

    //
    // Initialize the UART.
    //
    ConfigureUART();

    //
    // Print the welcome message to the terminal.
    //
    //UARTprintf("\033[2JMPU9150 Raw Example\n");

    //
    // Set the color to a purple approximation.
    //
    g_pui32Colors[RED] = 0x8000;
    g_pui32Colors[BLUE] = 0x8000;
    g_pui32Colors[GREEN] = 0x0000;

    //
    // Initialize RGB driver.
    //
    RGBInit(0);
    RGBColorSet(g_pui32Colors);
    RGBIntensitySet(0.5f);
    RGBEnable();

    //
    // The I2C3 peripheral must be enabled before use.
    //
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C3);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);

    //
    // Configure the pin muxing for I2C3 functions on port D0 and D1.
    //
    ROM_GPIOPinConfigure(GPIO_PD0_I2C3SCL);
    ROM_GPIOPinConfigure(GPIO_PD1_I2C3SDA);

    //
    // Select the I2C function for these pins.  This function will also
    // configure the GPIO pins pins for I2C operation, setting them to
    // open-drain operation with weak pull-ups.  Consult the data sheet
    // to see which functions are allocated per pin.
    //
    GPIOPinTypeI2CSCL(GPIO_PORTD_BASE, GPIO_PIN_0);
    ROM_GPIOPinTypeI2C(GPIO_PORTD_BASE, GPIO_PIN_1);

    //
    // Configure and Enable the GPIO interrupt. Used for INT signal from the
    // MPU9150
    //
    ROM_GPIOPinTypeGPIOInput(GPIO_PORTB_BASE, GPIO_PIN_2);
    GPIOIntEnable(GPIO_PORTB_BASE, GPIO_PIN_2);
    ROM_GPIOIntTypeSet(GPIO_PORTB_BASE, GPIO_PIN_2, GPIO_FALLING_EDGE);
    ROM_IntEnable(INT_GPIOB);

    //
    // Keep only some parts of the systems running while in sleep mode.
    // GPIOB is for the MPU9150 interrupt pin.
    // UART0 is the virtual serial port
    // TIMER0, TIMER1 and WTIMER5 are used by the RGB driver
    // I2C3 is the I2C interface to the ISL29023
    //
    ROM_SysCtlPeripheralClockGating(true);
    ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_GPIOB);
    ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_UART0);
    ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_TIMER0);
    ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_TIMER1);
    ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_I2C3);
    ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_WTIMER5);

    //
    // Enable interrupts to the processor.
    //
    ROM_IntMasterEnable();

    //
    // Initialize I2C3 peripheral.
    //
    I2CMInit(&g_sI2CInst, I2C3_BASE, INT_I2C3, 0xff, 0xff,
    ROM_SysCtlClockGet());

    //
    // Initialize the MPU9150 Driver.
    //
    MPU9150Init(&g_sMPU9150Inst, &g_sI2CInst, MPU9150_I2C_ADDRESS,
                MPU9150AppCallback, &g_sMPU9150Inst);

    //
    // Wait for transaction to complete
    //
    MPU9150AppI2CWait(__FILE__, __LINE__);

    //
    // Write application specifice sensor configuration such as filter settings
    // and sensor range settings.
    //
    g_sMPU9150Inst.pui8Data[0] = MPU9150_CONFIG_DLPF_CFG_94_98;
    g_sMPU9150Inst.pui8Data[1] = MPU9150_GYRO_CONFIG_FS_SEL_250;
    g_sMPU9150Inst.pui8Data[2] = (MPU9150_ACCEL_CONFIG_ACCEL_HPF_5HZ |
    MPU9150_ACCEL_CONFIG_AFS_SEL_2G);
    MPU9150Write(&g_sMPU9150Inst, MPU9150_O_CONFIG, g_sMPU9150Inst.pui8Data, 3,
                 MPU9150AppCallback, &g_sMPU9150Inst);

    //
    // Wait for transaction to complete
    //
    MPU9150AppI2CWait(__FILE__, __LINE__);

    //
    // Configure the data ready interrupt pin output of the MPU9150.
    //
    g_sMPU9150Inst.pui8Data[0] = MPU9150_INT_PIN_CFG_INT_LEVEL |
    MPU9150_INT_PIN_CFG_INT_RD_CLEAR |
    MPU9150_INT_PIN_CFG_LATCH_INT_EN;
    g_sMPU9150Inst.pui8Data[1] = MPU9150_INT_ENABLE_DATA_RDY_EN;
    MPU9150Write(&g_sMPU9150Inst, MPU9150_O_INT_PIN_CFG,
                 g_sMPU9150Inst.pui8Data, 2, MPU9150AppCallback,
                 &g_sMPU9150Inst);

    //
    // Wait for transaction to complete
    //
    MPU9150AppI2CWait(__FILE__, __LINE__);

    //
    // Initialize the DCM system. 50 hz sample rate.
    // accel weight = .2, gyro weight = .8, mag weight = .2
    //
    CompDCMInit(&g_sCompDCMInst, 1.0f / 50.0f, 0.2f, 0.6f, 0.2f);

    //
    // Enable blinking indicates config finished successfully
    //
    RGBBlinkRateSet(1.0f);

    ui32CompDCMStarted = 0;

    init_test();

    Delay_MicroSecond(3000000);

    while (1)
    {
        //
        // Go to sleep mode while waiting for data ready.
        //
        while (!g_vui8I2CDoneFlag)
        {
            ROM_SysCtlSleep();
        }

        //
        // Clear the flag
        //
        g_vui8I2CDoneFlag = 0;

        //
        // Get floating point version of the Accel Data in m/s^2.
        //
        MPU9150DataAccelGetFloat(&g_sMPU9150Inst, pfAccel, pfAccel + 1,
                                 pfAccel + 2);

        //
        // Get floating point version of angular velocities in rad/sec
        //
//        MPU9150DataGyroGetFloat(&g_sMPU9150Inst, pfGyro, pfGyro + 1,
//                                pfGyro + 2);

        //
        // Get floating point version of magnetic fields strength in tesla
        //
//        MPU9150DataMagnetoGetFloat(&g_sMPU9150Inst, pfMag, pfMag + 1,
//                                   pfMag + 2);

        //
        // Check if this is our first data ever.
        //
        if (ui32CompDCMStarted == 0)
        {
            //
            // Set flag indicating that DCM is started.
            // Perform the seeding of the DCM with the first data set.
            //
            ui32CompDCMStarted = 1;
            CompDCMMagnetoUpdate(&g_sCompDCMInst, pfMag[0], pfMag[1], pfMag[2]);
            CompDCMAccelUpdate(&g_sCompDCMInst, pfAccel[0], pfAccel[1],
                               pfAccel[2]);
            CompDCMGyroUpdate(&g_sCompDCMInst, pfGyro[0], pfGyro[1], pfGyro[2]);
            CompDCMStart(&g_sCompDCMInst);
        }
        else
        {
            //
            // DCM Is already started.  Perform the incremental update.
            //
//            CompDCMMagnetoUpdate(&g_sCompDCMInst, pfMag[0], pfMag[1],
//                                 pfMag[2]);
            CompDCMAccelUpdate(&g_sCompDCMInst, pfAccel[0], pfAccel[1],
                               pfAccel[2]);
//            CompDCMGyroUpdate(&g_sCompDCMInst, -pfGyro[0], -pfGyro[1],
//                              -pfGyro[2]);
            CompDCMUpdate(&g_sCompDCMInst);
        }

        //
        // Increment the skip counter.  Skip counter is used so we do not
        // overflow the UART with data.
        //
        g_ui32PrintSkipCounter++;
        if (g_ui32PrintSkipCounter >= PRINT_SKIP_COUNT)
        {
            //
            // Reset skip counter.
            //
            g_ui32PrintSkipCounter = 0;

            //
            // Get Euler data. (Roll Pitch Yaw)
            //
//            CompDCMComputeEulers(&g_sCompDCMInst, pfEulers, pfEulers + 1,
//                                 pfEulers + 2);

            //
            // Get Quaternions.
            //
//            CompDCMComputeQuaternion(&g_sCompDCMInst, pfQuaternion);

            //
            // convert mag data to micro-tesla for better human interpretation.
            //
//            pfMag[0] *= 1e6;
//            pfMag[1] *= 1e6;
//            pfMag[2] *= 1e6;

            //
            // Convert Eulers to degrees. 180/PI = 57.29...
            // Convert Yaw to 0 to 360 to approximate compass headings.
            //
//            pfEulers[0] *= 57.295779513082320876798154814105f;
//            pfEulers[1] *= 57.295779513082320876798154814105f;
//            pfEulers[2] *= 57.295779513082320876798154814105f;
//            if(pfEulers[2] < 0)
//            {
//                pfEulers[2] += 360.0f;
//            }

            //
            // Now drop back to using the data as a single array for the
            // purpose of decomposing the float into a integer part and a
            // fraction (decimal) part.
            //
            for (ui32Idx = 0; ui32Idx < 16; ui32Idx++)
            {
                //
                // Conver float value to a integer truncating the decimal part.
                //
                i32IPart[ui32Idx] = (int32_t) pfData[ui32Idx];

                //
                // Multiply by 1000 to preserve first three decimal values.
                // Truncates at the 3rd decimal place.
                //
                i32FPart[ui32Idx] = (int32_t) (pfData[ui32Idx] * 1000.0f);

                //
                // Subtract off the integer part from this newly formed decimal
                // part.
                //
                i32FPart[ui32Idx] = i32FPart[ui32Idx]
                        - (i32IPart[ui32Idx] * 1000);

                //
                // make the decimal part a positive number for display.
                //
                if (i32FPart[ui32Idx] < 0)
                {
                    i32FPart[ui32Idx] *= -1;
                }
            }

           /////////   Application code ///////////////////////////

            for (k = 0; k < 3; k++)                           /// Extract instantaneous X, Y ,Z acceleration values
            {
                buffer_A[k] = pfData[k];
            }
            X_present[w] = buffer_A[0];
            Y_present[w] = buffer_A[1];
            Z_present[w] = buffer_A[2];

            w++;

            UARTprintf("data ready\n");
            if (w == 10)
            {
                ROM_IntMasterDisable();                          /// Disable interrupt from MPU9150 so that the read values are not over written
                w = 0;
                int n = 10;
                int m = 11;
                UARTprintf("data ready\n");
                //////////////////   Compare with the Up gesture using DTW algo    //////////////////////////////
                DTW_X_up = dtw(X_present, X_up, n, m);
                DTW_Y_up = dtw(Y_present, Y_up, n, m);
                DTW_Z_up = dtw(Z_present, Z_up, n, m);
                DTW_up_tot = DTW_X_up + DTW_Y_up + DTW_Z_up;                   /// Get the total of X, Y and Z

                //////////////////   Compare with the Down gesture using DTW algo    //////////////////////////////
                DTW_X_down = dtw(X_present, X_down, n, m);
                DTW_Y_down = dtw(Y_present, Y_down, n, m);
                DTW_Z_down = dtw(Z_present, Z_down, n, m);
                DTW_down_tot = DTW_X_down + DTW_Y_down + DTW_Z_down;

                //////////////////   Compare with the Right gesture using DTW algo    //////////////////////////////
                DTW_X_right = dtw(X_present, X_right, n, m);
                DTW_Y_right = dtw(Y_present, Y_right, n, m);
                DTW_Z_right = dtw(Z_present, Z_right, n, m);
                DTW_right_tot = DTW_X_right + DTW_Y_right + DTW_Z_right;

                //////////////////   Compare with the Left gesture using DTW algo    //////////////////////////////
                DTW_X_left = dtw(X_present, X_left, n, m);
                DTW_Y_left = dtw(Y_present, Y_left, n, m);
                DTW_Z_left = dtw(Z_present, Z_left, n, m);
                DTW_left_tot = DTW_X_left + DTW_Y_left + DTW_Z_left;

                //////////////////   Compare with the Rest gesture using DTW algo    //////////////////////////////
                DTW_X_rest = dtw(X_present, X_rest, n, m);
                DTW_Y_rest = dtw(Y_present, Y_rest, n, m);
                DTW_Z_rest = dtw(Z_present, Z_rest, n, m);
                DTW_rest_tot = DTW_X_rest + DTW_Y_rest + DTW_Z_rest;


                ///////      Get the minimum of DTW_up_tot, DTW_down_tot, DTW_right_tot, DTW_left_tot, DTW_rest_tot ///////////
                min = fminf(DTW_up_tot, DTW_down_tot);
                min = fminf(min, DTW_right_tot);
                min = fminf(min, DTW_left_tot);
                min = fminf(min, DTW_rest_tot);

                if (min == DTW_up_tot)                           // PC4 - PWM  - PUL +VE   Up
                {                                               // DIR +VE - PE1 -Up
                    UARTprintf(" UP detected\n");
                    GPIO_PORTE_DATA_R = 0x00;                    // PE 1 = 0  For Up
                    PWM0_ENABLE_R = 0x40;
                    Delay_MicroSecond(1000000);
                    PWM0_ENABLE_R = 0x00;
                }
                else if (min == DTW_down_tot)              //// PC4 - PWM  - PUL +VE  -Down
                {                                         // DIR +VE - PE1 -Down
                    GPIO_PORTE_DATA_R = (1 << 1);                         // PE 1 = 1  For Down
                    UARTprintf(" Dejkkzsjdvk detected\n");
                    PWM0_ENABLE_R = 0x40;
                    Delay_MicroSecond(1000000);
                    PWM0_ENABLE_R = 0x00;
                }
                else if (min == DTW_right_tot)            // PE4 - PWM  - PUL +VE   Right
                {                                         // DIR +VE - PE2 -Right
                    UARTprintf(" Right detected\n");
                    GPIO_PORTE_DATA_R = 0x04;             // PE 2 = 1  For Down
                    PWM1_ENABLE_R = 0x04;
                    Delay_MicroSecond(1000000);
                    PWM1_ENABLE_R = 0x00;
                }
                else if (min == DTW_left_tot)             // PE4 - PWM  - PUL +VE   Left
                {                                        // DIR +VE - PE3 - Left
                    UARTprintf(" Left detected\n");
                    GPIO_PORTE_DATA_R = 0x00;            // PE 2 = 1  For Down
                    PWM1_ENABLE_R = 0x04;
                    Delay_MicroSecond(1000000);
                    PWM1_ENABLE_R = 0x00;
                }
                else
                {
                    UARTprintf(" Rest detected\n");
                }
                ROM_IntMasterEnable();
            }
            Delay_MicroSecond(20000);
        }
    }
}

