#include <string.h>
#include <inc/tm4c129encpdt.h>

#include "sca.h"

void init_uart();

void init_timer();

void uart_log(const char* inp, int len);

uint32_t get_timer_val();

char    key[100]    = "password";
int     key_len     = -1;

int8_t relu(int8_t inp) {
    if(inp < 0)
        return 0;
    else
        return inp;
}

int8_t relu_secure(int8_t inp) {
    int8_t mask = (inp & 0x80) >> 8;
    return mask && inp;
}

struct sca_context ctx;
char sca_inp_buffer[50];
char sca_meas_buffer[32];

int main(void) {
    key_len = strlen(key);

    init_uart();
    init_timer();

    // Initialize SCA and set uart log function as callback
    sca_init(&ctx, sca_inp_buffer, 50, SCA_UINT8, sca_meas_buffer, 32, SCA_UINT8);
    sca_register_uart_tx_callback(&ctx, uart_log);

    // All inputs are Q3.4
    int8_t inp;
    int8_t weight  = 0x10;
    int8_t bias    = 0x10;

    while(1) {
        if(sca_check_input_available(&ctx)) {
            inp = sca_get_next_inp(&ctx);

            start_sca();
            int8_t relu_inp = ((((int16_t)weight) * ((int16_t)inp)) >> 4) + bias;

            //int8_t out      = relu(relu_inp);
            int8_t out      = relu_secure(relu_inp);
            stop_sca();

            sca_add_meas(&ctx, relu_inp);
            sca_add_meas(&ctx, out);
            sca_set_time(&ctx, get_timer_val());
            sca_tx(&ctx);
            sca_reset(&ctx);
        }
    }

	return 0;
}

void keyset_parse(char* uart_buf, int uart_idx) {
    const char* keyset_command = "KEY:";
    int keyset_command_len = strlen(keyset_command);

    int idx = 0;

    while(idx < keyset_command_len && idx < uart_idx)
        if(uart_buf[idx] == keyset_command[idx])
            idx++;
        else
            return;

    for(int i = 0; i < 100; i++)
        key[i] = 0;

    while(idx < uart_idx) {
        key[idx - keyset_command_len] = uart_buf[idx];
        idx++;
    }

    key_len = uart_idx - keyset_command_len;

    const char* log_line = "Key was set.\n";
    uart_log(log_line, strlen(log_line));
}

void UART0_Int_Handler() {
    static char    uart_buf[100];       // String received from UART
    static short   uart_idx = 0;        // UART index
    char ch_rd;

    if(UART0_FR_R & 0x40)
        ch_rd = UART0_DR_R;
    else
        return;

    if(ch_rd == '\r') {
        int ret = sca_rx_parse(&ctx, uart_buf, uart_idx);
        if(ret == 0) {
            keyset_parse(uart_buf, uart_idx);
        }

        uart_idx = 0;
    }
    else {
        if(uart_idx >= 100)
            return;
        uart_buf[uart_idx] = ch_rd;
        uart_idx++;
    }
}

void init_uart() {
    SYSCTL_RCGCUART_R   |= 1;               // Enable UART0 clock
    SYSCTL_RCGCGPIO_R   |= 1;               // Enable PORTA clock

    for(int i = 0; i < 100; i++);

    UART0_CTL_R         = 0;

    GPIO_PORTA_AHB_AFSEL_R  |= 0x03;            // Set PA0, PA1 to alternate functions
    GPIO_PORTA_AHB_DEN_R    |= 0x03;            // Set PA0, PA1 to digital
    GPIO_PORTA_AHB_AMSEL_R  = 0;
    GPIO_PORTA_AHB_PCTL_R   |= 0x0011;          // Set PA0 to U0Rx, PA1 to U0Tx functions

    UART0_IBRD_R        = 8;                // Integer part of clock divider    = 8 (115200 baud)
    UART0_FBRD_R        = 44;               // Fractional part of clock divider = 0.6805 * 64 + 0.5 = 44.055 (115200 baud)

    UART0_CC_R          = 0;                // Use system clock

    UART0_LCRH_R        = 0x60;             // Line control : 1 stop bit, no interrupt, no parity, no FIFO

    UART0_CTL_R         = 0x0301;           // Enable UART Tx and Rx

    UART0_IM_R          = 0x10;             // Enable interrupt on rx
    NVIC_PRI1_R = (NVIC_PRI7_R & 0xFFFF1FFF) | 0x0000A000;  // Priority level 5
    NVIC_EN0_R  = NVIC_EN0_R | 0x00000020;
}

void init_timer() {
    SYSCTL_RCGCTIMER_R = 1;                         // Enable clock to Timer0
    for(int i = 0; i < 100; i++);

    TIMER0_CTL_R    = 0;                            // Disable timer
    TIMER0_CFG_R    = 0;                            // 32-bit mode,
    TIMER0_TAMR_R   = 0x00000011;                   // One-shot mode
    TIMER0_TAILR_R  = 0xFFFFFFFF;                   // Max count
    TIMER0_TAV_R    = 0;
}

void uart_send_byte(char inp) {
    while(UART0_FR_R & 0x20);
    UART0_DR_R = inp;
    while(UART0_FR_R & 0x20);
}

void uart_log(const char* inp, int len) {
    int idx = 0;
    while(inp[idx] != '\0' && idx < len) {
        uart_send_byte(inp[idx++]);
    }
}

// Disable interrupts and start timer0
void inline start_sca() {
    __asm ("    CPSID  I\n");
    TIMER0_TAV_R = 0;
    TIMER0_CTL_R    = 0x0003;
}

// Stop timer0 and enable interrupts
void inline stop_sca() {
    TIMER0_CTL_R = 0;
    __asm  ("    CPSIE  I\n");
}

uint32_t get_timer_val() {
    return TIMER0_TAV_R;
}
