#include <string.h>
#include <stdio.h>

#include "sca.h"

// Initialize the SCA context
void sca_init(
    struct sca_context* ctx,
    uint8_t* inp_buf,
    uint32_t inp_buf_len,
    enum sca_data_type inp_dtype,
    uint8_t* meas_buf,
    uint32_t meas_buf_len,
    enum sca_data_type meas_dtype
) {
    ctx->tx_keyword             = SCA_TX_KEYWORD;
    ctx->tx_keyword_len         = strlen(SCA_TX_KEYWORD);
    ctx->rx_keyword             = SCA_RX_KEYWORD;
    ctx->rx_keyword_len         = strlen(SCA_RX_KEYWORD);

    ctx->inputs.data            = inp_buf;
    ctx->inputs.type            = inp_dtype;
    ctx->inputs.count           = 0;
    ctx->inputs.len_bytes       = inp_buf_len;

    ctx->meas.data              = meas_buf;
    ctx->meas.type              = meas_dtype;
    ctx->meas.count             = 0;
    ctx->meas.len_bytes         = meas_buf_len;

    ctx->time                   = 0;

    ctx->input_idx              = 0;
    ctx->meas_idx               = 0;

    ctx->uart_tx_callback       = 0;

    ctx->rx_flag                = 0;
}

// Register callback for transmitting data by UART
void sca_register_uart_tx_callback(struct sca_context* ctx, void (*callback)(char*, int len)) {
    ctx->uart_tx_callback = callback;
}

// Reset input, timing, measurements
void sca_reset(struct sca_context* ctx) {
    ctx->inputs.count   = 0;
    ctx->meas.count     = 0;
    ctx->time           = 0;

    ctx->rx_flag        = 0;
    ctx->input_idx      = 0;
    ctx->meas_idx       = 0;
}

// Add an element to inputs array
void sca_add_input(struct sca_context* ctx, uint32_t inp) {
    switch(ctx->inputs.type) {
    case SCA_UINT8:
    case SCA_CHAR:
        if(ctx->inputs.count >= ctx->inputs.len_bytes)         return;

        (ctx->inputs.data)[ctx->inputs.count]              = (uint8_t)inp;

        break;
    case SCA_UINT16:
        if(2 * ctx->inputs.count >= ctx->inputs.len_bytes)     return;

        ((uint16_t*)(ctx->inputs.data))[ctx->inputs.count] = (uint16_t)inp;

        break;
    case SCA_UINT32:
        if(4 * ctx->inputs.count >= ctx->inputs.len_bytes)     return;

        ((uint32_t*)(ctx->inputs.data))[ctx->inputs.count] = (uint32_t)inp;

        break;
    }

    ctx->inputs.count++;
    return;
}

// Add an element to measurements array
void sca_add_meas(struct sca_context* ctx, uint32_t inp) {
    switch(ctx->meas.type) {
        case SCA_UINT8:
        case SCA_CHAR:
            if(ctx->meas.count >= ctx->meas.len_bytes)         return;

            (ctx->meas.data)[ctx->meas.count]              = (uint8_t)inp;

            break;
        case SCA_UINT16:
            if(2 * ctx->meas.count >= ctx->meas.len_bytes)     return;

            ((uint16_t*)(ctx->meas.data))[ctx->meas.count] = (uint16_t)inp;

            break;
        case SCA_UINT32:
            if(4 * ctx->meas.count >= ctx->meas.len_bytes)     return;

            ((uint32_t*)(ctx->meas.data))[ctx->meas.count] = (uint32_t)inp;

            break;
    }

    ctx->meas.count++;
    return;
}

// Set measurement data from string
void sca_set_meas_string(struct sca_context* ctx, char* inp, uint32_t len) {
    if(len > ctx->meas.len_bytes || ctx-> meas.type != SCA_CHAR)
        return;

    for(int i = 0; i < len; i++)
        ctx->meas.data[i] = inp[i];

    ctx->meas.count = len;
}

// Parse string received from UART and add to the inputs array
uint8_t sca_rx_parse(struct sca_context* ctx, char* str, uint32_t strlen) {
    int j;

    for(j = 0; j < ctx->rx_keyword_len; j++) {
        if(j >= strlen)
            return 0;

        if(ctx->rx_keyword[j] != str[j])
            return 0;
    }

    ctx->rx_flag = 1;

    uint32_t    temp = 0;

    switch(ctx->inputs.type) {
        // For char type input, just push the received string directly
        case SCA_CHAR:
            while(j < strlen) {
                sca_add_input(ctx, str[j++]);
            }

            break;
        // For unsigned integer inputs, convert from hex
        case SCA_UINT8:
        case SCA_UINT16:
        case SCA_UINT32:
            for(;j < strlen; j++) {
                if(str[j] >= '0' && str[j] <= '9')
                    temp = (temp << 4) + (str[j] - '0');
                else if(str[j] == 'A' || str[j] == 'a')
                    temp = (temp << 4) + 0xA;
                else if(str[j] == 'B' || str[j] == 'b')
                    temp = (temp << 4) + 0xB;
                else if(str[j] == 'C' || str[j] == 'c')
                    temp = (temp << 4) + 0xC;
                else if(str[j] == 'D' || str[j] == 'd')
                    temp = (temp << 4) + 0xD;
                else if(str[j] == 'E' || str[j] == 'e')
                    temp = (temp << 4) + 0xE;
                else if(str[j] == 'F' || str[j] == 'f')
                    temp = (temp << 4) + 0xF;
                else if(str[j] == ',') {
                    sca_add_input(ctx, temp);
                    temp = 0;
                } else
                    return 0;
            }
            sca_add_input(ctx, temp);

            break;
    }

    return 1;
}

// Check if inputs were received
uint8_t sca_check_received(struct sca_context* ctx) {
    return ctx->rx_flag;
}

// Check if more inputs are available
uint8_t sca_check_input_available(struct sca_context* ctx) {
    return (ctx->input_idx) < (ctx->inputs.count);
}

// Get the next input
uint32_t sca_get_next_inp(struct sca_context* ctx) {
    if(!sca_check_input_available(ctx))
        return 0;

    switch(ctx->inputs.type) {
    case SCA_CHAR:
    case SCA_UINT8:
        return ctx->inputs.data[ctx->input_idx++];
    case SCA_UINT16:
        return ((uint16_t*)ctx->inputs.data)[ctx->input_idx++];
    case SCA_UINT32:
        return ((uint32_t*)ctx->inputs.data)[ctx->input_idx++];
    }
}

// Set recording time
uint32_t sca_set_time(struct sca_context* ctx, uint32_t time) {
    ctx->time = time;
}

// Sends measurement and output
void sca_tx(struct sca_context* ctx) {
    char buf[100];

    char meas_buf[100] = "";
    uint32_t idx = 0;

    switch(ctx->meas.type) {
        case SCA_CHAR:
            for(int i = 0; i < ctx->meas.count && i < 100; i++)
                meas_buf[i] = ctx->meas.data[i];
            break;
        case SCA_UINT8:
            if(ctx->meas.count <= 0)
                break;

            idx = sprintf(meas_buf, "%X", (uint8_t)ctx->meas.data[0]);

            for(int i = 1; i < ctx->meas.count;i++)
                idx += sprintf(meas_buf + idx, ",%X", ((uint8_t*)ctx->meas.data)[i]);

            break;
        case SCA_UINT16:
            if(ctx->meas.count <= 0)
                break;

            idx = sprintf(meas_buf, "%X", (uint16_t)ctx->meas.data[0]);

            for(int i = 1; i < ctx->meas.count;i++)
                idx += sprintf(meas_buf + idx, ",%X", ((uint16_t*)ctx->meas.data)[i]);

            break;
        case SCA_UINT32:
            if(ctx->meas.count <= 0)
                break;

            idx = sprintf(meas_buf, "%X", (uint32_t)ctx->meas.data[0]);

            for(int i = 1; i < ctx->meas.count;i++)
                idx += sprintf(meas_buf + idx, ",%X", ((uint32_t*)ctx->meas.data)[i]);

            break;
    }

    int len = sprintf(buf, "%s%s:%d\n", ctx->tx_keyword, meas_buf, ctx->time);

    ctx->uart_tx_callback(buf, len);
}
