/*****************************************************
 *
 * 256-bit ECC implementation
 * Short Weierstrass curves
 * Any 256-bit prime field
 *
 * Big Integer adopted from ARM mbedTLS:
 * https://github.com/ARMmbed/mbedtls
 *
 * Last Modified: 2022 May 3
 *
 */

#include <stdio.h>
#include <stdint.h>

// Big Integer
struct ecclib_mpi
{
    uint32_t p[8];  // limbs
};

void print_str_uart(char* str);
void conv_hex_char(uint32_t val);
// Print X
void ecclib_mpi_print( struct ecclib_mpi *X );

// X = 0
void ecclib_mpi_init( struct ecclib_mpi *X );

// X = Y
void ecclib_mpi_copy( struct ecclib_mpi *X, struct ecclib_mpi *Y );

// Swap X & Y
void ecclib_mpi_swap( struct ecclib_mpi *X, struct ecclib_mpi *Y );

// X = z
void ecclib_mpi_lset( struct ecclib_mpi *X, uint32_t z );

// X = (X << 1)
void ecclib_mpi_shift_l( struct ecclib_mpi *X );

// X = (X >> 1)
void ecclib_mpi_shift_r( struct ecclib_mpi *X );

// (carry, X) = (A + B)
uint32_t ecclib_mpi_add_mpi( struct ecclib_mpi *X, struct ecclib_mpi *A, struct ecclib_mpi *B );

// (borrow, X) = (A - B)
uint32_t ecclib_mpi_sub_mpi( struct ecclib_mpi *X, struct ecclib_mpi *A, struct ecclib_mpi *B );

// Compare X & Y
int ecclib_mpi_cmp_mpi( struct ecclib_mpi *X, struct ecclib_mpi *Y );

// Compare X & z
int ecclib_mpi_cmp_int( struct ecclib_mpi *X, uint32_t z );

// X = (A + B) mod p
void ecclib_fp_add( struct ecclib_mpi *X, struct ecclib_mpi *A, struct ecclib_mpi *B, struct ecclib_mpi *prime );

// X = (A - B) mod p
void ecclib_fp_sub( struct ecclib_mpi *X, struct ecclib_mpi *A, struct ecclib_mpi *B, struct ecclib_mpi *prime );

// X = (A * B) mod p
void ecclib_fp_mul( struct ecclib_mpi *X, struct ecclib_mpi *A, struct ecclib_mpi *B, struct ecclib_mpi *prime );

// X = (A / B) mod p
void ecclib_fp_div( struct ecclib_mpi *X, struct ecclib_mpi *A, struct ecclib_mpi *B, struct ecclib_mpi *prime );

// Elliptic Curve Point
struct ecclib_ecp
{
    struct ecclib_mpi X;    // x-coordinate
    struct ecclib_mpi Y;    // y-coordinate
    uint32_t inf;           // flag for point at infinity
};

// Print P
void ecclib_ecp_print( struct ecclib_ecp *P );

// P = Inf
void ecclib_ecp_init( struct ecclib_ecp *P );

// R = P
void ecclib_ecp_copy( struct ecclib_ecp *R, struct ecclib_ecp *P );

// Check if P1 lies on y**2 = x**3 + (a * x) + b
int ecclib_ecp_point_check( struct ecclib_ecp *P1, struct ecclib_mpi *param_a, struct ecclib_mpi *param_b, struct ecclib_mpi *prime );

// P3 = 2(P1)
void ecclib_ecp_affine_dbl( struct ecclib_ecp *P3, struct ecclib_ecp *P1,
                            struct ecclib_mpi *param_a, struct ecclib_mpi *prime );

// P3 = P1 + P2
void ecclib_ecp_affine_add( struct ecclib_ecp *P3, struct ecclib_ecp *P1, struct ecclib_ecp *P2,
                            struct ecclib_mpi *param_a, struct ecclib_mpi *prime );

// Q = k(P)
int ecclib_ecp_affine_scalar_mul( struct ecclib_ecp *Q, struct ecclib_ecp *P, struct ecclib_mpi *k,
                                  struct ecclib_mpi *param_a, struct ecclib_mpi *param_b,
                                  struct ecclib_mpi *order, struct ecclib_mpi *prime );
