/*****************************************************
 *
 * 256-bit ECC implementation
 * Short Weierstrass curves
 * Any 256-bit prime field
 *
 * Big Integer adopted from ARM mbedTLS:
 * https://github.com/ARMmbed/mbedtls
 *
 * Last Modified: 2022 May 3
 *
 */

#include "ECCLib.h"
#include <string.h>
#include "UART_Driver.h"

const char HEX [16] = {'0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'};
unsigned char debug_msg[10];
unsigned char enter[3]={'\r','\n'};
unsigned char X[4]={'X',':',' '};
unsigned char Y[4]={'Y',':',' '};

void print_str_uart(char* str)
{
    int len;
    len=strlen(str);
    UART_Write_Blocking(
           0,           /* port identifier */
           len,           /* max number of bytes that should be written */
           str);        /* buffer to be used to write into FIFO */
}
void conv_hex_char(uint32_t val)
{
    debug_msg[0]=HEX[(val >> 28) & 0x0F];
    debug_msg[1]=HEX[(val >> 24) & 0x0F];
    debug_msg[2]=HEX[(val >> 20) & 0x0F];
    debug_msg[3]=HEX[(val >> 16) & 0x0F];
    debug_msg[4]=HEX[(val >> 12) & 0x0F];
    debug_msg[5]=HEX[(val >> 8) & 0x0F];
    debug_msg[6]=HEX[(val >> 4) & 0x0F];
    debug_msg[7]=HEX[(val >> 0) & 0x0F];
}

void ecclib_mpi_print( struct ecclib_mpi *X )
{
    size_t i;

    for( i = 8; i > 0; i-- )
    {
        conv_hex_char(X->p[i-1]);

        UART_Write_Blocking(
               0,           /* port identifier */
               8,           /* max number of bytes that should be written */
               debug_msg);        /* buffer to be used to write into FIFO */
    }
    UART_Write_Blocking(
           0,           /* port identifier */
           2,           /* max number of bytes that should be written */
           enter);        /* buffer to be used to write into FIFO */
}


void ecclib_mpi_init( struct ecclib_mpi *X )
{
    size_t i;

    for( i = 0; i < 8; i++ )
        X->p[i] = 0;
}

void ecclib_mpi_copy( struct ecclib_mpi *X, struct ecclib_mpi *Y )
{
    size_t i;

    for( i = 0; i < 8; i++ )
        X->p[i] = Y->p[i];
}

void ecclib_mpi_swap( struct ecclib_mpi *X, struct ecclib_mpi *Y )
{
    struct ecclib_mpi T;

    ecclib_mpi_copy( &T,  X );
    ecclib_mpi_copy(  X,  Y );
    ecclib_mpi_copy(  Y, &T );
}

void ecclib_mpi_lset( struct ecclib_mpi *X, uint32_t z )
{
    size_t i;

    for( i = 1; i < 8; i++ )
        X->p[i] = 0;
    X->p[0] = z;
}

void ecclib_mpi_shift_l( struct ecclib_mpi *X )
{
    size_t i;
    uint32_t r0 = 0, r1;

    for( i = 0; i < 8; i++ )
    {
        r1 = X->p[i] >> 31;
        X->p[i] <<= 1;
        X->p[i] |= r0;
        r0 = r1;
    }
}

void ecclib_mpi_shift_r( struct ecclib_mpi *X )
{
    size_t i;
    uint32_t r0 = 0, r1;

    for( i = 8; i > 0; i-- )
    {
        r1 = X->p[i - 1] << 31;
        X->p[i - 1] >>= 1;
        X->p[i - 1] |= r0;
        r0 = r1;
    }
}

uint32_t ecclib_mpi_add_mpi( struct ecclib_mpi *X, struct ecclib_mpi *A, struct ecclib_mpi *B )
{
    size_t i;
    uint32_t carry = 0, sum = 0;

    for( i = 0; i < 8; i++ )
    {
        sum = A->p[i] + B->p[i] + carry;
        if( sum != A->p[i] )
            carry = (sum < A->p[i]);
        X->p[i] = sum;
    }
    return carry;
}

uint32_t ecclib_mpi_sub_mpi( struct ecclib_mpi *X, struct ecclib_mpi *A, struct ecclib_mpi *B )
{
    size_t i;
    uint32_t borrow = 0, diff = 0;

    for( i = 0; i < 8; i++ )
    {
        diff = A->p[i] - B->p[i] - borrow;
        if( diff != A->p[i] )
            borrow = (diff > A->p[i]);
        X->p[i] = diff;
    }
    return borrow;
}

int ecclib_mpi_cmp_mpi( struct ecclib_mpi *X, struct ecclib_mpi *Y )
{
    size_t i;

    for( i = 8; i > 0; i-- )
    {
        if( X->p[i-1] > Y->p[i-1] )
            return 1;
        if( X->p[i-1] < Y->p[i-1] )
            return -1;
    }
    return 0;
}

int ecclib_mpi_cmp_int( struct ecclib_mpi *X, uint32_t z )
{
    size_t i;

    for( i = 8; i > 1; i-- )
    {
        if ( X->p[i-1] > 0 )
            return 1;
    }
    if( X->p[0] > z )
        return 1;
    else if ( X->p[0] < z )
        return -1;
    else
        return 0;
}

void ecclib_fp_add( struct ecclib_mpi *X, struct ecclib_mpi *A, struct ecclib_mpi *B, struct ecclib_mpi *prime )
{
    struct ecclib_mpi S0, D1;
    uint32_t carry, borrow;

    carry  = ecclib_mpi_add_mpi( &S0, A, B );
    borrow = ecclib_mpi_sub_mpi( &D1, &S0, prime );

    if( carry == 1 || borrow == 0 )
        ecclib_mpi_copy( X, &D1 );
    else
        ecclib_mpi_copy( X, &S0 );
}

void ecclib_fp_sub( struct ecclib_mpi *X, struct ecclib_mpi *A, struct ecclib_mpi *B, struct ecclib_mpi *prime )
{
    struct ecclib_mpi D0, S1;
    uint32_t borrow, carry;

    borrow = ecclib_mpi_sub_mpi( &D0, A, B );
    carry  = ecclib_mpi_add_mpi( &S1, &D0, prime );

    if( borrow == 1 )
        ecclib_mpi_copy( X, &S1 );
    else
        ecclib_mpi_copy( X, &D0 );
}

void ecclib_fp_mul( struct ecclib_mpi *X, struct ecclib_mpi *A, struct ecclib_mpi *B, struct ecclib_mpi *prime )
{
    struct ecclib_mpi Z, T;
    int i;

    ecclib_mpi_copy( &T, A );
    ecclib_mpi_lset( &Z, 0 );

    for( i = 256; i > 0; i-- )
    {
        ecclib_fp_add( &Z, &Z, &Z, prime );
        if( ( T.p[7] >> 31 ) == 1 )
        {
            ecclib_fp_add( &Z, &Z, B, prime );
        }
        ecclib_mpi_shift_l( &T );
    }
    
    ecclib_mpi_copy( X, &Z );
}

void ecclib_fp_div( struct ecclib_mpi *X, struct ecclib_mpi *A, struct ecclib_mpi *B, struct ecclib_mpi *prime )
{
    struct ecclib_mpi U, V, T1, T2;
    uint32_t borrow, carry;

    ecclib_mpi_copy( &U, B );
    ecclib_mpi_copy( &V, prime );
    ecclib_mpi_copy( &T1, A );
    ecclib_mpi_lset( &T2, 0 );

    while( ecclib_mpi_cmp_int( &U, 0 ) != 0 )
    {
        while( ( U.p[0] & 1 ) == 0 )
        {
            ecclib_mpi_shift_r( &U );

            if( ( T1.p[0] & 1 ) == 0 )
            {
                ecclib_mpi_shift_r( &T1 );
            }
            else
            {
                carry = ecclib_mpi_add_mpi( &T1, &T1, prime );
                ecclib_mpi_shift_r( &T1 );
                T1.p[7] |= (carry << 31);
            }
        }

        while( ( V.p[0] & 1 ) == 0 )
        {
            ecclib_mpi_shift_r( &V );

            if( ( T2.p[0] & 1 ) == 0 )
            {
                ecclib_mpi_shift_r( &T2 );
            }
            else
            {
                carry = ecclib_mpi_add_mpi( &T2, &T2, prime );
                ecclib_mpi_shift_r( &T2 );
                T2.p[7] |= (carry << 31);
            }
        }

        if( ecclib_mpi_cmp_mpi( &U, &V ) >= 0 )
        {
            ecclib_mpi_sub_mpi( &U, &U, &V );
            borrow = ecclib_mpi_sub_mpi( &T1, &T1, &T2 );
            if( borrow == 1 )
                ecclib_mpi_add_mpi( &T1, &T1, prime );
        }
        else
        {
            ecclib_mpi_sub_mpi( &V, &V, &U );
            borrow = ecclib_mpi_sub_mpi( &T2, &T2, &T1 );
            if( borrow == 1 )
                ecclib_mpi_add_mpi( &T2, &T2, prime );
        }
    }

    ecclib_mpi_copy( X, &T2 );
}

void ecclib_ecp_print( struct ecclib_ecp *P )
{
    UART_Write_Blocking(
           0,           /* port identifier */
           3,           /* max number of bytes that should be written */
           X);        /* buffer to be used to write into FIFO */
    ecclib_mpi_print( &P->X );
    UART_Write_Blocking(
           0,           /* port identifier */
           3,           /* max number of bytes that should be written */
           Y);        /* buffer to be used to write into FIFO */
    ecclib_mpi_print( &P->Y );
}


void ecclib_ecp_init( struct ecclib_ecp *P )
{
    ecclib_mpi_init( &P->X );
    ecclib_mpi_init( &P->Y );
    P->inf = 1;
}

void ecclib_ecp_copy( struct ecclib_ecp *R, struct ecclib_ecp *P )
{
    ecclib_mpi_copy( &R->X, &P->X );
    ecclib_mpi_copy( &R->Y, &P->Y );
    R->inf = P->inf;
}

int ecclib_ecp_point_check( struct ecclib_ecp *P1, struct ecclib_mpi *param_a, struct ecclib_mpi *param_b, struct ecclib_mpi *prime )
{
    struct ecclib_mpi T1, T2;

    if( P1->inf == 1 )
    {
        return 0;
    }
    else
    {
        ecclib_fp_mul( &T1, &P1->X, &P1->X, prime );   // T1 = x1**2
        ecclib_fp_mul( &T1, &T1, &P1->X, prime );      // T1 = x1**3
        ecclib_fp_mul( &T2, &P1->X, param_a, prime );  // T2 = a * x
        ecclib_fp_add( &T2, &T2, &T1, prime );         // T2 = x1**3 + (a * x)
        ecclib_fp_add( &T2, &T2, param_b, prime );     // T2 = x1**3 + (a * x) + b

        ecclib_fp_mul( &T1, &P1->Y, &P1->Y, prime );   // T1 = y1**2

        if( ecclib_mpi_cmp_mpi( &T1, &T2 ) == 0 )
        {
            return 0;
        }
        else
        {
            return -1;
        }
    }
}

void ecclib_ecp_affine_dbl( struct ecclib_ecp *P3, struct ecclib_ecp *P1,
                            struct ecclib_mpi *param_a, struct ecclib_mpi *prime )
{
    struct ecclib_mpi T1, T2, T3;

    if( P1->inf == 1 )
    {
        P3->inf = 1;
    }
    else if( ecclib_mpi_cmp_int( &P1->Y, 0 ) == 0 )
    {
        P3->inf = 1;
    }
    else 
    {
        ecclib_fp_mul( &T1, &P1->X, &P1->X, prime );   // T1 = x1**2
        ecclib_fp_add( &T2, &T1, &T1, prime);          // T2 = 2 * x1**2
        ecclib_fp_add( &T2, &T2, &T1, prime);          // T2 = 3 * x1**2
        ecclib_fp_add( &T2, &T2, param_a, prime);      // T2 = (3 * x1**2) + a
        ecclib_fp_add( &T1, &P1->Y, &P1->Y, prime);    // T1 = 2 * y1
        ecclib_fp_div( &T2, &T2, &T1, prime);          // T2 = ((3 * x1**2) + a) / (2 * y1) = lambda

        ecclib_fp_mul( &T1, &T2, &T2, prime);          // T1 = lambda**2
        ecclib_fp_sub( &T1, &T1, &P1->X, prime);       // T1 = lambda**2 - x1
        ecclib_fp_sub( &T3, &T1, &P1->X, prime);       // T3 = lambda**2 - (2 * x1) = x3

        ecclib_fp_sub( &T1, &P1->X, &T3, prime);       // T1 = x1 - x3
        ecclib_fp_mul( &T2, &T2, &T1, prime);          // T2 = lambda * (x1 - x3)
        ecclib_fp_sub( &T2, &T2, &P1->Y, prime);       // T2 = lambda * (x1 - x3) - y1

        ecclib_mpi_copy( &P3->X, &T3 );                // x3 = T3
        ecclib_mpi_copy( &P3->Y, &T2 );                // y3 = T2

        P3->inf = 0;
    }
}

void ecclib_ecp_affine_add( struct ecclib_ecp *P3, struct ecclib_ecp *P1, struct ecclib_ecp *P2,
                            struct ecclib_mpi *param_a, struct ecclib_mpi *prime )
{
    struct ecclib_mpi T1, T2, T3;

    if( P1->inf == 1 )
    {
        ecclib_mpi_copy( &P3->X, &P2->X );
        ecclib_mpi_copy( &P3->Y, &P2->Y );
        P3->inf = P2->inf;
    }
    else if( P2->inf == 1 )
    {
        ecclib_mpi_copy( &P3->X, &P1->X );
        ecclib_mpi_copy( &P3->Y, &P1->Y );
        P3->inf = P1->inf;
    }
    else if( ecclib_mpi_cmp_mpi( &P1->X, &P2->X ) == 0 )
    {
        if( ecclib_mpi_cmp_mpi( &P1->Y, &P2->Y ) == 0 )
        {
            ecclib_ecp_affine_dbl( P3, P1, param_a, prime);
        }
        else
        {
            P3->inf = 1;
        }
    }
    else 
    {
        ecclib_fp_sub( &T1, &P2->X, &P1->X, prime );   // T1 = x2 - x1
        ecclib_fp_sub( &T2, &P2->Y, &P1->Y, prime);    // T2 = y2 - y1
        ecclib_fp_div( &T2, &T2, &T1, prime);          // T2 = (y2 - y1) / (x2 - x1) = lambda

        ecclib_fp_mul( &T1, &T2, &T2, prime);          // T1 = lambda**2
        ecclib_fp_sub( &T1, &T1, &P1->X, prime);       // T1 = lambda**2 - x1
        ecclib_fp_sub( &T3, &T1, &P2->X, prime);       // T3 = lambda**2 - x1 - x2 = x3

        ecclib_fp_sub( &T1, &P1->X, &T3, prime);       // T1 = x1 - x3
        ecclib_fp_mul( &T2, &T2, &T1, prime);          // T2 = lambda * (x1 - x3)
        ecclib_fp_sub( &T2, &T2, &P1->Y, prime);       // T2 = lambda * (x1 - x3) - y1

        ecclib_mpi_copy( &P3->X, &T3 );                // x3 = T3
        ecclib_mpi_copy( &P3->Y, &T2 );                // y3 = T2

        P3->inf = 0;
    }
}

int ecclib_ecp_affine_scalar_mul( struct ecclib_ecp *Q, struct ecclib_ecp *P, struct ecclib_mpi *k,
                                  struct ecclib_mpi *param_a, struct ecclib_mpi *param_b,
                                  struct ecclib_mpi *order, struct ecclib_mpi *prime )
{
    struct ecclib_mpi T;
    struct ecclib_ecp R;
    int i;
    //k=x(sk), p=G, have to do Q=p*k, i.e., H=xG
    if( ecclib_mpi_cmp_mpi( k, order ) >= 0 )
    {
        return -1;
    }
    if( ecclib_ecp_point_check( P, param_a, param_b, prime ) )
    {
        return -1;
    }

    R.inf = 1;

    ecclib_mpi_copy( &T, k );

    for( i = 0; i < 256; i++ )
    {
        ecclib_ecp_affine_dbl( &R, &R, param_a, prime );

        if( ( T.p[7] >> 31 ) == 1 )
        {
            ecclib_ecp_affine_add( &R, &R, P, param_a, prime);
        }

        ecclib_mpi_shift_l( &T );
    }

    ecclib_ecp_copy( Q, &R );
    return 0;
}








