/*
 * Peripheral_Init.c
 *
 *      Author: surya
 */

/* Contains functions which can be used to initialise given peripherals and hardware interfaces

*/

#include <./inc/tm4c123gh6pm.h>
#include "Peripheral_Init.h"
#include <stdbool.h>



/* Main Timer - A 32 bit timer which can count max upto 268.43s
                1 sec count
                Uses the 32 bit timer Timer0
                Enable/Disable Timer in Main program
*/
void Timer_Main_init()
{
    SYSCTL_RCGCWTIMER_R |= 0x01; // enable clock to WTimer0
    WTIMER0_CTL_R = 0x00;        // Disable timer for initialisation
    WTIMER0_CFG_R = 0x4;       // 32bit mode
    WTIMER0_TAMR_R = 0x2;        // Periodic mode
    WTIMER0_IMR_R = 0x1;         // Enable Timer time out interrupt mask
    WTIMER0_TAILR_R = CYCLES_1SEC;  // count for 1s
    NVIC_PRI23_R = (NVIC_PRI23_R & 0xFF1FFFFF) | 0x00A00000; /*  priority 5 */
    NVIC_EN2_R = 0x40000000;        /*  Enable interrupt 94 in NVIC */
}


/* Timer to refresh 7seg display - A 16bit timer which can count max upto 4ms
                                   1ms count
                                   uses Timer A of Timer1
                                   Enable disable in Main Program
*/
void Timer_7seg_init()
{
    SYSCTL_RCGCTIMER_R |= 0x02; // enable clock to Timer1
    TIMER1_CTL_R = 0x00;        // Disable timer for initialisation
    TIMER1_CFG_R = 0x004;       // 16bit mode
    TIMER1_TAMR_R = 0x2;        // Periodic mode
    TIMER1_IMR_R = 0x1;         // Enable Timer A time out interrupt mask
    TIMER1_TAILR_R = CYCLES_1MSEC;  // count for 1ms
    NVIC_PRI5_R = (NVIC_PRI5_R & 0xFFFF1FFF) | 0x0000A000; /*  priority 5 */
    NVIC_EN0_R = 0x00200000;        /*  Enable interrupt 21 in NVIC */

}

/* Timer to wait for LCD delay - A 16 bit timer which can count upto 4ms
                                 2 delays 1.64ms and 40us
                                 uses Timer A of Timer2 in one shot mode
                                 Enable disable and set count in Main Program
*/
void Timer_LCD_init()
{
    SYSCTL_RCGCTIMER_R |= 0x04; // enable clock to Timer2
    TIMER2_CTL_R = 0x00;        // Disable timer for initialisation
    TIMER2_CFG_R = 0x000;       // 32bit mode
    TIMER2_TAMR_R = 0x2;        // Periodic  mode
    TIMER2_IMR_R = 0x1;         // Enable Timer A time out interrupt mask
    TIMER2_TAILR_R = CYCLES_LCD3;
    NVIC_PRI5_R = (NVIC_PRI5_R & 0x1FFFFFFF) | 0xA0000000; /*  priority 5 */
    NVIC_EN0_R = 0x00800000;        /*  Enable interrupt 23 in NVIC */
}

/* Timer for LED blinking delay - A 32 bit timer with delay of 0.5s
                                  uses Timer A of Timer3 in periodic mode
                                  Enable disable in Main Program
*/
void Timer_LED_init()
{
    SYSCTL_RCGCTIMER_R |= 0x08; // enable clock to Timer3
    TIMER3_CTL_R = 0x00;        // Disable timer for initialisation
    TIMER3_CFG_R = 0x000;       // 32 bit mode
    TIMER3_TAMR_R = 0x2;        // Periodic  mode
    TIMER3_IMR_R = 0x1;         // Enable Timer A time out interrupt mask
    TIMER3_TAILR_R = CYCLES_LED;// count 0.5s
    NVIC_PRI8_R = (NVIC_PRI8_R & 0x1FFFFFFF) | 0xA0000000; /*  priority 5 */
    NVIC_EN1_R = 0x00000008;        /*  Enable interrupt 35 in NVIC */
}

/* Timer for data logging  - A 24 bit timer with delay of 800ms
                                  uses Timer A of Timer4 in one shot mode
                                  Enable disable in Main Program
*/
void Timer_log_0_8s()
{
    SYSCTL_RCGCTIMER_R |= 0x10; // enable clock to Timer4
    TIMER4_CTL_R = 0x00;        // Disable timer for initialisation
    TIMER4_CFG_R = 0x000;       // 16 bit mode
    TIMER4_TAMR_R = 0x2;        // one shot  mode
    TIMER4_IMR_R = 0x1;         // Enable Timer A time out interrupt mask
    TIMER4_TAILR_R = 0x3D09000;    // count 0.8s
    //TIMER4_TAPR_R = 0xC3;
    NVIC_PRI17_R = (NVIC_PRI17_R & 0xFF1FFFFF) | 0x00A00000; /*  priority 5 */
    NVIC_EN2_R = 0x00000040;        /*  Enable interrupt 70 in NVIC */
}

/* Timer for data logging  - A 24 bit timer with delay of 1s
                                  uses Timer A of Timer5 in one shot mode
                                  Enable disable in Main Program
*/
void Timer_log_1s()
{
    SYSCTL_RCGCTIMER_R |= 0x20; // enable clock to Timer5
    TIMER5_CTL_R = 0x00;        // Disable timer for initialisation
    TIMER5_CFG_R = 0x000;       // 16 bit mode
    TIMER5_TAMR_R = 0x2;        // one shot  mode
    TIMER5_IMR_R = 0x1;         // Enable Timer A time out interrupt mask
    TIMER5_TAILR_R = 0x3D09000;    // count 1s
    //TIMER5_TAPR_R = 0xF4;
    NVIC_PRI23_R = (NVIC_PRI23_R & 0xFFFFFF1F) | 0x000000A0; /*  priority 5 */
    NVIC_EN2_R = (NVIC_EN2_R & ~10000000) | 0x10000000;        /*  Enable interrupt 92 in NVIC */
}

/* Main Timer - A 32 bit timer which can count max upto 268.43s
                1 sec count
                Uses the 32 bit timer Timer0
                Enable/Disable Timer in Main program
*/
void Timer_log_1_2s()
{
    SYSCTL_RCGCWTIMER_R |= 0x02; // enable clock to WTimer1
    WTIMER1_CTL_R = 0x00;        // Disable timer for initialisation
    WTIMER1_CFG_R = 0x4;       // 32bit mode
    WTIMER1_TAMR_R = 0x2;        // One shot mode
    WTIMER1_IMR_R = 0x1;         // Enable Timer time out interrupt mask
    WTIMER1_TAILR_R = 0x3D09000;  // count for 1.2s
    NVIC_PRI24_R = (NVIC_PRI24_R & 0xFFFFFF1F) | 0x000000A0; /*  priority 5 */
    NVIC_EN3_R = 0x00000001;        /*  Enable interrupt 96 in NVIC */
}

/* Main Timer - A 32 bit timer which can count max upto 268.43s
                1 sec count
                Uses the 32 bit timer Timer0
                Enable/Disable Timer in Main program
*/
void Timer_log_10s()
{
    SYSCTL_RCGCWTIMER_R |= 0x04; // enable clock to WTimer2
    WTIMER2_CTL_R = 0x00;        // Disable timer for initialisation
    WTIMER2_CFG_R = 0x4;       // 32bit mode
    WTIMER2_TAMR_R = 0x2;        // Periodic mode
    WTIMER2_IMR_R = 0x1;         // Enable Timer time out interrupt mask
    WTIMER2_TAILR_R = 0x1C9C3800;  // count for 10s
    NVIC_PRI24_R = (NVIC_PRI24_R & 0xFF1FFFFF) | 0x00A00000; /*  priority 5 */
    NVIC_EN3_R = 0x00000004;        /*  Enable interrupt 98 in NVIC */
}

/* Initialise LEDS on PF3-1 and SW's on PF4 and PF0
   Also arm voiderrupts for the switches 
   Enable in NVIC in main program
*/
void LED_SW_init()
{
    SYSCTL_RCGC2_R |= 0x20;     // enable clock to Port F
    GPIO_PORTF_LOCK_R = 0x4C4F434B;   // unlock Port F
    GPIO_PORTF_CR_R = 0x1F;      // enable changes to PF4-0
    GPIO_PORTF_AMSEL_R = 0x00;   // disable analog on Port F
    GPIO_PORTF_DIR_R = 0x0E;     // PF4 AND PF0 in , PF3-0 out
    GPIO_PORTF_PUR_R = 0x11;     // enable pull up on PF0 and PF4
    GPIO_PORTF_DEN_R = 0x1F;     // PF4-0 digital enable

    // voiderrupt enable for sw1 and sw2
    GPIO_PORTF_IS_R &= ~0x11;    // PF4 and PF0 are edge sensitive
    GPIO_PORTF_IBE_R &= ~0x11;   // not on both edged
    GPIO_PORTF_IEV_R &= ~0x11;   // falling edge event
    GPIO_PORTF_ICR_R &= 0x11;    // clear flag4 and flag0
    GPIO_PORTF_IM_R  |= 0x11;    // arm voiderrupt on PF4 AND PF0

    NVIC_PRI7_R = (NVIC_PRI7_R & 0xFF1FFFFF) | 0x00A00000; /*  priority 5 */

    NVIC_EN0_R = 0x40000000;        /*  Enable interrupt 30 in NVIC */
}

/* Enable PWM1_0 on pin PD0 and PA2 for direction*/
void pwm_init()
{
    SYSCTL_RCGC2_R |= 0x09; // enable clock to Port D and Port A
    SYSCTL_RCGCPWM_R |= 2;     // enable clock to PWM1
    SYSCTL_RCC_R |= 0x2E0000;  // sysclk/64 i.e 4us time period

    GPIO_PORTD_AFSEL_R |= 0x01; // alternate function for PD0
    GPIO_PORTD_PCTL_R = 0x05;   // M1PWM0

    //GPIO_PORTD_DIR_R = (GPIO_PORTD_DIR_R & ~0x01) + 0x01; // dir as output
    GPIO_PORTA_DIR_R = (GPIO_PORTA_DIR_R & ~0x0C) + 0x0C;

    GPIO_PORTD_DEN_R = (GPIO_PORTD_DEN_R & ~0x01) + 0x01; // digital enable
    GPIO_PORTA_DEN_R = (GPIO_PORTA_DEN_R & ~0x0C) + 0x0C;



    /* PWM Configuration */
    PWM1_0_CTL_R = 0;
    PWM1_0_LOAD_R = 50;        // 1Khz frequency
    PWM1_0_CMPA_R = 25;
    PWM1_0_GENA_R = 0x8C;

    GPIO_PORTA_DATA_R = (GPIO_PORTA_DATA_R & ~0x0C) + 0x08;

}

/* Enable Pins PB7-0 and PA7-4 as output digital enable for 7 segment display
   and LCD 
*/
void seg7_init()
{
    SYSCTL_RCGC2_R |= 0x03;      // enable clock to Port A and B
    GPIO_PORTB_DIR_R = 0xFF;     // PB7-0 are out
    GPIO_PORTA_DIR_R = (GPIO_PORTA_DIR_R & ~0xF0) + 0xF0;   // PA4-7 are out
    GPIO_PORTB_DEN_R = 0xFF;     // PB7-0 Digital enable
    GPIO_PORTA_DEN_R = (GPIO_PORTA_DEN_R & ~0xF0) + 0xF0;   // PA4-7 digital enable
}

/*  7 seg display functions */

// mapping from Integer to Hex for 7segment display
unsigned int DispMap(int n)
{
    unsigned int valueB;
    switch(n)
           {
           case 0:
               valueB = 0x3F;
               break;
           case 1:
               valueB = 0x06;
               break;
           case 2:
               valueB = 0x5B;
               break;
           case 3:
               valueB = 0x4F;
               break;
           case 4:
               valueB = 0x66;
               break;
           case 5:
               valueB = 0x6D;
               break;
           case 6:
               valueB = 0x7D;
               break;
           case 7:
               valueB = 0x07;
               break;
           case 8:
               valueB = 0x7F;
               break;
           case 9:
               valueB = 0x6F;
               break;
           default:
               valueB = 0x40;
           }
    return valueB;
}

// Display 1's place
void seg7Disp1(int n)
{
    unsigned int valueB;
    valueB = DispMap(n);

   GPIO_PORTA_DATA_R = (GPIO_PORTA_DATA_R & ~0xF0 ) + 0x10;
   GPIO_PORTB_DATA_R = valueB;
}

//Display 10's place
void seg7Disp10(int n)
{

    unsigned int valueB;
    valueB = DispMap(n) + 0x80;

   GPIO_PORTA_DATA_R = (GPIO_PORTA_DATA_R & ~0xF0 ) + 0x20;
   GPIO_PORTB_DATA_R = valueB;
}

//Display 100's place
void seg7Disp100(int n)
{
    unsigned int valueB;
    valueB = DispMap(n);

   GPIO_PORTA_DATA_R = (GPIO_PORTA_DATA_R & ~0xF0 ) + 0x40;
   GPIO_PORTB_DATA_R = valueB;
}

// Display 1000's place
void seg7Disp1000(int n)
{
    unsigned int valueB;
    valueB = DispMap(n);

   GPIO_PORTA_DATA_R = (GPIO_PORTA_DATA_R & ~0xF0 ) + 0x80;
   GPIO_PORTB_DATA_R = valueB;
}


